<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RolesAndPermissionsSeeder extends Seeder
{
    public function run(): void
    {
        // Reset cached roles and permissions
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // Define permissions for both web and API
        $permissions = [
            // Dashboard access
            'access superadmin dashboard',
            'access admin dashboard',
            'access doctor dashboard',
            'access patient dashboard',
            'access staff dashboard',

            // Role management
            'view superadmin roles',
            'view doctor roles',
            'view patient roles',
            'view staff roles',

            // User management
            'list users',
            'create users',
            'edit users',
            'delete users',
            'manage user status',

            // Clinic management
            'list clinics',
            'create clinics',
            'edit clinics',
            'delete clinics',
            'manage clinic staff',
            'view clinics',

            // Doctor management
            'list doctors',
            'create doctors',
            'edit doctors',
            'delete doctors',
            'view doctor profile',
            'manage doctor schedules',

            // Patient management
            'list patients',
            'create patients',
            'edit patients',
            'delete patients',
            'view patient profile',

            // Staff management
            'manage staff',
            'assign staff roles',
            'manage staff permissions',
            'list staff',
            'create staff',
            'edit staff',
            'view staff',
            'delete staff',

            // Appointment management
            'list appointments',
            'create appointments',
            'edit appointments',
            'delete appointments',
            'approve appointments',
            'reject appointments',
            'view appointment details',

            // Schedule management
            'manage schedules',
            'view schedules',
            'edit doctor availability',

            // Department management
            'manage departments',
            'assign departments',

            // Plans management
            'manage subscription plans',
            'view subscription plans',
            'create subscription plans',
            'edit subscription plans',
            'manage usage types',
            'view usage types',
            'create usage types',
            'edit usage types',
            'manage plan features',
            'view plan features',
            'create plan features',
            'edit plan features',

            // Financial permissions
            'manage invoices',
            'create invoices',
            'view invoices',
            'manage payments',
            'process payments',
            'manage expenses',
            'view financial reports',

            // Medical records
            'manage medical records',
            'view medical records',
            'create prescriptions',
            'view prescriptions',
            'edit medical history',

            // Reports
            'generate reports',
            'view reports',
            'export reports',
            'view clinic reports',

            // System settings
            'manage system settings',
            'view active devices',
            'logout active devices',
            'manage roles',
            'assign permissions',
            'view audit logs',
            'configure clinic settings'
        ];

        // Create permissions for all guards
        $guards = ['web', 'api', 'superadmin', 'doctor' , 'patient'];
        foreach ($guards as $guard) {
            foreach ($permissions as $permission) {
                Permission::firstOrCreate(['name' => $permission, 'guard_name' => $guard]);
            }
        }

        // Define role hierarchy with their appropriate guards
        $roleConfigs = [
            'superadmin' => [
                'guards' => ['superadmin', 'api','web'], // SuperAdmin uses ONLY superadmin and api guards
                'permissions' => $permissions,
                'group' => 'superadmin_group',
                'children' => [
                    'superadminstaff' => [
                        'permissions' => [
                            'access superadmin dashboard',
                            'access admin dashboard',
                            'list users',
                            'create users',
                            'edit users',
                            'list clinics',
                            'view clinics',
                            'create clinics',
                            #'edit clinics',
                            #'manage staff',
                            #'list appointments',
                            #'create appointments',
                            #'edit appointments',
                            #'manage schedules',
                            #'view reports',
                            #'view invoices',
                            #'view financial reports',
                            #'list doctors',
                            #'create doctors',
                            #'edit doctors',
                            #'view doctor profile',
                            #'view subscription plans',
                            #'view usage types',
                            #'view plan features'
                        ],
                        'group' => 'superadmin_group'
                    ]
                ]
            ],
            'doctor' => [
                'guards' => ['doctor','web', 'api'],
                'permissions' => [
                    'access doctor dashboard',
                    'view doctor roles',
                    'list doctors',
                    'create doctors',
                    'edit doctors',
                    'delete doctors',
                    'view doctor profile',
                    'manage doctor schedules',
                    'list patients',
                    'create patients',
                    'delete patients',
                    'edit patients',
                    'view patient profile',
                    'list appointments',
                    'approve appointments',
                    'reject appointments',
                    'manage doctor schedules',
                    'create prescriptions',
                    'manage medical records',
                    'view reports',
                    'view invoices',
                    'list clinics',
                    'view clinics',
                    'edit doctor availability',
                    'manage clinic staff',
                    'manage staff',
                    'list staff',
                    'create staff',
                    'edit staff',
                    'view staff',
                    'delete staff',
                ],
                'group' => 'doctor_group',
                'children' => [
                    'doctorstaff' => [
                        'permissions' => [
                            'access staff dashboard',
                            'access doctor dashboard',
                            #'list patients',
                            #'view patient profile',
                            #'list appointments',
                            #'create appointments',
                            #'edit appointments',
                            #'view schedules',
                            #'view medical records',
                            #'view prescriptions',
                            #'view invoices',
                            #'list clinics',
                            #'view clinics',
                        ],
                        'group' => 'doctor_group'
                    ]
                ]
            ],
            'patient' => [
                'guards' => ['web','patient','api'], // Patient uses web/api guards
                'permissions' => [
                    'access patient dashboard',
                    'view patient roles',
                    'list doctors',
                    'view doctor profile',
                    'list appointments',
                    'create appointments',
                    'view medical records',
                    'view prescriptions',
                    'view invoices',
                    'process payments',
                    'list clinics',
                    'view clinics'
                ],
                'group' => 'patient_group'
            ]
        ];

        // Create roles and assign permissions
        foreach ($roleConfigs as $roleName => $roleData) {
            foreach ($roleData['guards'] as $guard) {
                // Create or update the main role
                $role = Role::updateOrCreate(
                    ['name' => $roleName, 'guard_name' => $guard],
                    ['group' => $roleData['group']]
                );
                $guardPermissions = Permission::where('guard_name', $guard)
                    ->whereIn('name', $roleData['permissions'])
                    ->get();

                $role->syncPermissions($guardPermissions);

                if (isset($roleData['children'])) {
                    foreach ($roleData['children'] as $childName => $childData) {
                        $childRole = Role::updateOrCreate(
                            ['name' => $childName, 'guard_name' => $guard],
                            [
                                'group' => $childData['group'],
                                'parent_role' => $roleName
                            ]
                        );

                        $childGuardPermissions = Permission::where('guard_name', $guard)
                            ->whereIn('name', $childData['permissions'])
                            ->get();

                        $childRole->syncPermissions($childGuardPermissions);
                    }
                }
            }
        }

        // Create enum-compatible role aliases
        $this->createRoleAliases();
    }

    protected function createRoleAliases(): void
    {
        $aliases = [
            'primary_doctor' => [
                'parent' => 'doctor',
                'guards' => ['doctor', 'web', 'api'], // Use doctor guard primarily
                'permissions' => [
                    'configure clinic settings',
                    'view clinic reports',
                    'edit clinics',
                    'assign staff roles',
                    'manage staff',
                    'list staff',
                    'create staff',
                    'edit staff',
                    'view staff',
                    'delete staff',
                ],
                'group' => 'doctor_group'
            ],
            'sub_doctor' => [
                'parent' => 'doctor',
                'guards' => ['doctor', 'web', 'api'], 
                'permissions' => [],
                'group' => 'doctor_group'
            ]
        ];

        foreach ($aliases as $alias => $data) {
            foreach ($data['guards'] as $guard) {
                $parent = Role::where('name', $data['parent'])
                    ->where('guard_name', $guard)
                    ->first();

                if ($parent) {
                    $role = Role::updateOrCreate(
                        ['name' => $alias, 'guard_name' => $guard],
                        [
                            'group' => $data['group'],
                            'parent_role' => $data['parent']
                        ]
                    );

                    // Get parent permissions
                    $permissions = $parent->permissions->pluck('name')->toArray();

                    // Merge with alias-specific permissions    
                    $allPermissions = array_unique(array_merge(
                        $permissions,
                        $data['permissions']
                    ));

                    $role->syncPermissions(Permission::whereIn('name', $allPermissions)
                        ->where('guard_name', $guard)
                        ->get());
                }
            }
        }
    }
}