<?php

namespace App\Models;

use App\Models\Clinic;
use App\Models\PatientClinic;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class Patient extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'address',
        'street_address',
        'city',
        'state',
        'postal_code',
        'country',
        'phone',
        'date_of_birth',
        'gender',
        'profile_image',
        'medical_history',
        'verified_at',
        'verified_by',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'verified_at' => 'timestamp',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function verifiedBy()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function clinics()
    {
        return $this->belongsToMany(Clinic::class, 'patient_clinics', 'patient_id', 'clinic_id')
            ->using(PatientClinic::class)
            ->withPivot([
                'doctor_id',
                'status',
                'approval_status',
                'added_by',
                'approved_at',
                'notes'
            ]);
    }

        public function getProfileImageUrlAttribute()
    {
        if ($this->profile_image) {
            return Storage::url($this->profile_image);
        }
        return null;
    }

    public function hasProfileImage()
    {
        return $this->profile_image && Storage::disk('public')->exists($this->profile_image);
    }

    public function getProfileImageOrDefaultAttribute()
    {
        if ($this->hasProfileImage()) {
            return $this->profile_image_url;
        }
        return asset('images/default-avatar.png');
    }

    public function patientClinics()
    {
        return $this->hasMany(PatientClinic::class);
    }

    public function doctors()
    {
        return $this->belongsToMany(User::class, 'patient_clinics', 'patient_id', 'doctor_id')
            ->withPivot(['clinic_id', 'status', 'approval_status']);
    }

        public function getDateOfBirthFormatted()
    {
        return $this->date_of_birth ? $this->date_of_birth->format('Y-m-d') : null;
    }
}