<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\Http;

class VideoConference extends Component
{
    // Constants
    public $API_BASE_URL = "https://api.videosdk.live";

    // Meeting state
    public $meeting = null;
    public $token = '';
    public $meetingId = '';
    public $joinName = '';
    public $joinMeetingCode = '';
    public $isMeetingJoined = false;
    public $isMeetingCreated = false;

    // Device states
    public $currentMic = null;
    public $currentCamera = null;
    public $currentPlayback = null;
    public $cameraPermissionAllowed = true;
    public $microphonePermissionAllowed = true;

    // Media states
    public $micEnabled = false;
    public $webCamEnabled = false;
    public $screenShareOn = false;
    public $isRecording = false;

    // UI states
    public $showParticipants = false;
    public $showChat = false;
    public $totalParticipants = 0;
    public $chatMessages = [];
    public $newChatMessage = '';
    public $raiseHandMessage = '';
    public $showRaiseHandAlert = false;

    // Device lists
    public $cameras = [];
    public $microphones = [];
    public $playbackDevices = [];

    protected $listeners = [
        'refreshDevices',
        'updateNetworkStats',
        'deviceChanged',
        'meetingJoined',
        'participantJoined',
        'participantLeft',
        'streamEnabled',
        'streamDisabled',
        'recordingStarted',
        'recordingStopped',
        'presenterChanged',
        'chatMessageReceived',
        'raiseHand'
    ];

    public function mount()
    {
        $this->getToken();
    }

    public function getToken()
    {
        $response = Http::withHeaders([
            'Authorization' => env('VIDEOSDK_SECRET'),
            'Content-Type' => 'application/json'
        ])->post('https://api.videosdk.live/v2/token', [
            'apikey' => env('VIDEOSDK_API_KEY')
        ]);

        if ($response->successful()) {
            $this->token = $response->json()['token'];
        }
    }

    public function joinMeeting($createNew = false)
    {
        if ($createNew) {
            $this->createNewMeeting();
        } else {
            $this->validateMeeting();
        }
    }

    public function createNewMeeting()
    {
        $response = Http::withHeaders([
            'Authorization' => $this->token,
            'Content-Type' => 'application/json'
        ])->post($this->API_BASE_URL . '/v2/rooms');

        if ($response->successful()) {
            $this->meetingId = $response->json()['roomId'];
            $this->isMeetingCreated = true;
            $this->isMeetingJoined = true;
            $this->dispatchBrowserEvent('start-meeting', [
                'token' => $this->token,
                'meetingId' => $this->meetingId,
                'name' => $this->joinName ?: 'Host'
            ]);
        }
    }

    public function validateMeeting()
    {
        if (empty($this->joinMeetingCode) || empty($this->joinName)) {
            return;
        }

        $response = Http::withHeaders([
            'Authorization' => $this->token,
        ])->get($this->API_BASE_URL . "/v2/rooms/validate/{$this->joinMeetingCode}");

        if ($response->successful() && $response->json()['roomId'] === $this->joinMeetingCode) {
            $this->meetingId = $this->joinMeetingCode;
            $this->isMeetingJoined = true;
            $this->dispatchBrowserEvent('start-meeting', [
                'token' => $this->token,
                'meetingId' => $this->meetingId,
                'name' => $this->joinName
            ]);
        }
    }

    public function toggleMic()
    {
        $this->micEnabled = !$this->micEnabled;
        $this->dispatchBrowserEvent('toggle-mic', ['enable' => $this->micEnabled]);
    }

    public function toggleWebCam()
    {
        $this->webCamEnabled = !$this->webCamEnabled;
        $this->dispatchBrowserEvent('toggle-webcam', ['enable' => $this->webCamEnabled]);
    }

    public function toggleScreenShare()
    {
        $this->screenShareOn = !$this->screenShareOn;
        $this->dispatchBrowserEvent('toggle-screen-share', ['enable' => $this->screenShareOn]);
    }

    public function toggleRecording()
    {
        $this->isRecording = !$this->isRecording;
        $this->dispatchBrowserEvent('toggle-recording', ['enable' => $this->isRecording]);
    }

    public function raiseHand()
    {
        $this->raiseHandMessage = 'You have raised your hand';
        $this->showRaiseHandAlert = true;
        $this->dispatchBrowserEvent('raise-hand');
        
        $this->dispatchBrowserEvent('hide-alert', ['timeout' => 2000]);
    }

    public function sendChatMessage()
    {
        if (!empty($this->newChatMessage)) {
            $this->dispatchBrowserEvent('send-chat-message', ['message' => $this->newChatMessage]);
            $this->chatMessages[] = [
                'sender' => 'You',
                'message' => $this->newChatMessage,
                'timestamp' => now()->format('H:i'),
                'isLocal' => true
            ];
            $this->newChatMessage = '';
        }
    }

    public function copyMeetingCode()
    {
        $this->dispatchBrowserEvent('copy-meeting-code', ['code' => $this->meetingId]);
    }

    public function leaveMeeting()
    {
        $this->dispatchBrowserEvent('leave-meeting');
        $this->resetMeetingState();
    }

    public function endMeeting()
    {
        $this->dispatchBrowserEvent('end-meeting');
        $this->resetMeetingState();
    }

    private function resetMeetingState()
    {
        $this->meetingId = '';
        $this->isMeetingJoined = false;
        $this->isMeetingCreated = false;
        $this->totalParticipants = 0;
        $this->chatMessages = [];
    }

    public function render()
    {
        return view('livewire.video-conference');
    }
}