<?php

namespace App\Livewire\Doctor\Profile;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\DoctorProfile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class DoctorCertificatesComponent extends Component
{
    use WithFileUploads;

    public $tempCertificates = [];
    public $uploadedCertificates = [];

    protected $rules = [
        'tempCertificates.*' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
    ];

    public function mount()
    {
        $doctorProfile = Auth::guard('doctor')->user()->doctorProfile;
        $this->uploadedCertificates = json_decode($doctorProfile->certificates, true) ?? [];
    }

    public function removeCertificate($index)
    {
        $certificate = $this->uploadedCertificates[$index];

        if (isset($certificate['path']) && Storage::disk('public')->exists($certificate['path'])) {
            Storage::disk('public')->delete($certificate['path']);
        }

        unset($this->uploadedCertificates[$index]);
        $this->uploadedCertificates = array_values($this->uploadedCertificates);
        $this->saveCertificates();
    }

    public function save()
    {
        $this->validate();

        if (!empty($this->tempCertificates)) {
            $this->uploadCertificates();
        }

        $this->tempCertificates = [];
        notyf()->success('Certificates updated successfully.');
    }

    protected function uploadCertificates()
    {
        $doctorProfile = Auth::guard('doctor')->user()->doctorProfile;
        $newCertificates = [];

        foreach ($this->tempCertificates as $certificate) {
            // Store with original filename in a structured path
            $path = $certificate->storeAs(
                'doctor/certificates/' . $doctorProfile->id,
                $certificate->getClientOriginalName(),
                'public'
            );
            
            $newCertificates[] = [
                'name' => $certificate->getClientOriginalName(),
                'path' => $path,
                'uploaded_at' => now()->toDateTimeString(),
            ];
        }

        $this->uploadedCertificates = array_merge($this->uploadedCertificates, $newCertificates);
        $this->saveCertificates();
    }

    protected function saveCertificates()
    {
        $doctorProfile = Auth::guard('doctor')->user()->doctorProfile;
        $doctorProfile->update([
            'certificates' => json_encode($this->uploadedCertificates)
        ]);
        
        // Refresh the uploaded certificates from database
        $this->uploadedCertificates = json_decode($doctorProfile->fresh()->certificates, true) ?? [];
    }

    public function render()
    {
        return view('livewire.doctor.profile.doctor-certificates-component');
    }
}