<?php

namespace App\Livewire\Doctor\Doctor;

use App\Models\User;
use App\Models\Clinic;
use App\Models\DoctorClinic;
use App\Models\DoctorProfile;
use App\Models\Doctor;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class DoctorList extends Component
{
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $searchTerm = '';
    public $statusFilter = 'all';
    public $specializationFilter = 'all';
    public $clinicFilter = 'all';
    public $perPage = 10;
    public $sortField = 'name';
    public $sortDirection = 'asc';

    protected $queryString = [
        'searchTerm' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'specializationFilter' => ['except' => 'all'],
        'clinicFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
        'sortField' => ['except' => 'name'],
        'sortDirection' => ['except' => 'asc'],
    ];

    public function updatingSearchTerm()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }

    public function updatingSpecializationFilter()
    {
        $this->resetPage();
    }

    public function updatingClinicFilter()
    {
        $this->resetPage();
    }

    public function updatingPerPage()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function toggleStatus($id)
    {
        if (auth()->id() == $id) {
            notyf()->error('You cannot change status your own account.');
            return;
        }

        Gate::authorize('edit doctors');
        $user = User::findOrFail($id);
        $clinic = $user->clinicAssociations()->where('clinic_id' , session('current_clinic_id'))->firstOrFail();
        $clinic->status = $clinic->status === 'active' ? 'inactive' : 'active';
        $clinic->save();

        session()->flash('message', 'Doctor Clinic status updated successfully.');
    }

    public function deleteDoctor($id)
    {
        Gate::authorize('delete doctors');

        // Prevent self-deletion
        if (auth()->id() == $id) {
            notyf()->error('You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        // Verify the user is actually a doctor
        if (!$user->isDoctor()) {
            notyf()->error('The selected user is not a doctor.');
            return;
        }

        $clinic_id = session('current_clinic_id');

        if (!$clinic_id) {
            notyf()->error('No clinic selected.');
            return;
        }

        // Detach from clinic
        $user->clinics()->detach($clinic_id);

        // Additional cleanup if needed
        $user->clinicAssociations()->where('clinic_id', $clinic_id)->delete();

        notyf()->success('Doctor removed from clinic successfully.');
        // If you need to refresh any Livewire data
        $this->dispatch('refreshDoctorsList');
    }
    
    public function mount()
    {
        Gate::authorize('list doctors');
    }
    public function render()
    {
        $authUser = auth()->guard('doctor')->user();

        // Check if user has permission to view this list
        if (!$authUser->hasRole('primary_doctor') || !$authUser->hasPrimaryClinic()) {
            return view('errors.403');
        }

        // Base query for doctors associated with the primary clinic
        $query = DoctorClinic::with([
            'doctor.doctorProfile',
            'clinic',
            'doctor.clinics' => function ($q) {
                $q->where('is_primary', true);
            }
        ])
            ->where('clinic_id', $authUser->primaryClinics[0]->id);
        // Apply search filter
        if (!empty($this->searchTerm)) {
            $searchTerm = '%' . $this->searchTerm . '%';
            $query->whereHas('doctor', function ($q) use ($searchTerm) {
                $q->where('name', 'like', $searchTerm)
                    ->orWhere('email', 'like', $searchTerm)
                    ->orWhere('phone_number', 'like', $searchTerm);
            });
        }

        // Apply status filter
        if ($this->statusFilter !== 'all') {
            $query->where('status', $this->statusFilter);
        }

        // Apply specialization filter
        if ($this->specializationFilter !== 'all') {
            $query->whereHas('doctor.doctorProfile', function ($q) {
                $q->where('specialization', $this->specializationFilter);
            });
        }

        // Apply clinic filter
        if ($this->clinicFilter !== 'all') {
            $query->where('clinic_id', $this->clinicFilter);
        }

        // Handle sorting
        if ($this->sortField === 'name') {
            $query->join('users', 'doctor_clinic.doctor_id', '=', 'users.id')
                ->orderBy('users.name', $this->sortDirection)
                ->select('doctor_clinic.*');
        } elseif ($this->sortField === 'status') {
            $query->orderBy('status', $this->sortDirection);
        }

        // Paginate results
        $doctorClinics = $query->paginate($this->perPage);

        // Transform results to match view expectations
        $users = $doctorClinics->map(function ($doctorClinic) {
            $user = $doctorClinic->doctor ?: new User();
            
            $clinic = $user->clinicAssociations()
                        ->where('clinic_id', session('current_clinic_id'))
                        ->firstOrFail();
            
            $user->clinic_status = $clinic->status;
            return $user;
        });

        // Get filter options
        $specializations = DoctorProfile::select('specialization')
            ->distinct()
            ->whereNotNull('specialization')
            ->orderBy('specialization')
            ->pluck('specialization')
            ->toArray();

        $clinics = Clinic::select('id', 'name')
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        return view('livewire.doctor.doctor.doctor-list', [
            'users' => new \Illuminate\Pagination\LengthAwarePaginator(
                $users,
                $doctorClinics->total(),
                $doctorClinics->perPage(),
                $doctorClinics->currentPage(),
                ['path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()],
            ),
            'specializations' => $specializations,
            'clinics' => $clinics,
            'doctorClinic'=>$doctorClinics,
        ]);
    }
}
