<?php

namespace App\Livewire\Doctor\Doctor;

use Livewire\Component;
use App\Models\User;
use App\Models\Clinic;
use App\Models\DoctorClinic;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\Validate;

class DoctorClinicManager extends Component
{
    public $doctorId;
    public $selectedDoctorId;
    public $selectedClinicId;
    public $isPrimary = false;
    public $status = 'active';
    
    #[Validate('nullable|numeric|min:0')]
    public $inPersonFee;
    
    #[Validate('nullable|numeric|min:0')]
    public $videoFee;
    
    #[Validate('nullable|numeric|min:0')]
    public $homeVisitFee;
    
    public $showForm = false;
    public $editingAssociationId = null;
    
    protected $rules = [
        'selectedDoctorId' => 'required|exists:users,id',
        'selectedClinicId' => 'required|exists:clinics,id',
        'inPersonFee' => 'nullable|numeric|min:0',
        'videoFee' => 'nullable|numeric|min:0',
        'homeVisitFee' => 'nullable|numeric|min:0',
    ];

    public function mount($doctorId = null)
    {
        $this->selectedDoctorId = $doctorId;
        $this->doctorId = $doctorId;
    }

    public function updatedSelectedDoctorId()
    {
        $this->doctorId = $this->selectedDoctorId;
        $this->resetForm();
        $this->showForm = false;
    }

    public function render()
    {
        // Get all doctors using Spatie roles
        $doctors = User::role(['doctor', 'primary_doctor', 'sub_doctor'])
                      ->orderBy('name')
                      ->get();

        $doctor = null;
        $availableClinics = collect();
        $doctorAssociations = collect();

        if ($this->doctorId) {
            $doctor = User::with(['clinics', 'ownedClinics'])->find($this->doctorId);
            
            if ($doctor) {
                $availableClinics = Clinic::whereNotIn('id', 
                    $doctor->clinics->pluck('id')
                )->get();
                
                $doctorAssociations = DoctorClinic::with(['clinic', 'addedBy'])
                    ->where('doctor_id', $this->doctorId)
                    ->get();
            }
        }

        return view('livewire.doctor.doctor.doctor-clinic-manager', [
            'doctors' => $doctors,
            'doctor' => $doctor,
            'availableClinics' => $availableClinics,
            'doctorAssociations' => $doctorAssociations
        ]);
    }

    public function showAddForm()
    {
        if (!$this->doctorId) {
            session()->flash('error', 'Please select a doctor first.');
            return;
        }
        
        $this->resetForm();
        $this->showForm = true;
    }

    public function editAssociation($associationId)
    {
        $association = DoctorClinic::find($associationId);
        
        if ($association && $association->doctor_id == $this->doctorId) {
            $this->editingAssociationId = $associationId;
            $this->selectedClinicId = $association->clinic_id;
            $this->isPrimary = $association->is_primary;
            $this->status = $association->status;
            $this->inPersonFee = $association->in_person_fee;
            $this->videoFee = $association->video_fee;
            $this->homeVisitFee = $association->home_visit_fee;
            $this->showForm = true;
        }
    }

    public function save()
    {
        $this->validate();

        if (!$this->doctorId) {
            session()->flash('error', 'Please select a doctor first.');
            return;
        }

        try {
            DB::transaction(function () {
                if ($this->isPrimary) {
                    DoctorClinic::where('doctor_id', $this->doctorId)
                        ->where('is_primary', true)
                        ->update(['is_primary' => false]);
                }

                $data = [
                    'doctor_id' => $this->doctorId,
                    'clinic_id' => $this->selectedClinicId,
                    'is_primary' => $this->isPrimary,
                    'status' => $this->status,
                    'in_person_fee' => $this->inPersonFee,
                    'video_fee' => $this->videoFee,
                    'home_visit_fee' => $this->homeVisitFee,
                    'added_by' => auth()->id(),
                    'approval_status' => 'approved'
                ];

                if ($this->editingAssociationId) {
                    DoctorClinic::where('id', $this->editingAssociationId)
                        ->update($data);
                    session()->flash('message', 'Clinic association updated successfully!');
                } else {
                    DoctorClinic::create($data);
                    session()->flash('message', 'Doctor added to clinic successfully!');
                }
            });

            $this->resetForm();
            $this->showForm = false;
        } catch (\Exception $e) {
            session()->flash('error', 'Error: ' . $e->getMessage());
        }
    }

    public function removeAssociation($associationId)
    {
        try {
            $association = DoctorClinic::find($associationId);
            
            if ($association && $association->doctor_id == $this->doctorId) {
                $association->delete();
                session()->flash('message', 'Clinic association removed successfully!');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'Error removing association: ' . $e->getMessage());
        }
    }

    public function setPrimary($associationId)
    {
        try {
            DB::transaction(function () use ($associationId) {
                DoctorClinic::where('doctor_id', $this->doctorId)
                    ->update(['is_primary' => false]);
                
                DoctorClinic::where('id', $associationId)
                    ->where('doctor_id', $this->doctorId)
                    ->update(['is_primary' => true]);
            });
            
            session()->flash('message', 'Primary clinic updated successfully!');
        } catch (\Exception $e) {
            session()->flash('error', 'Error updating primary clinic: ' . $e->getMessage());
        }
    }

    public function cancel()
    {
        $this->resetForm();
        $this->showForm = false;
    }

    private function resetForm()
    {
        $this->selectedClinicId = null;
        $this->isPrimary = false;
        $this->status = 'active';
        $this->inPersonFee = null;
        $this->videoFee = null;
        $this->homeVisitFee = null;
        $this->editingAssociationId = null;
        $this->resetErrorBag();
    }
}