<?php

namespace App\Livewire\Admin\SubscriptionPlanWizard;

use App\Models\AddOn;
use App\Models\Feature;
use Livewire\Component;
use App\Models\PlanLimit;
use App\Models\UsageRate;
use App\Models\UsageType;
use App\Models\PlanFeature;
use App\Models\SubscriptionPlan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class Wizard extends Component
{
    public $currentStep = 1;
    public $highestStep = 1;
    public $totalSteps = 5;
    public $editingPlanId = null;
    public $isEditing = false;

    public $planData = [
        'basic' => [],
        'limits' => [],
        'features' => [],
        'rates' => [],
        'is_combo' => [],
    ];

    protected $listeners = [
        'goToStep',
        'updatePlanData',
        'savePlan'
    ];

    public function mount($planId = null)
    {
        if ($planId) {
            $this->editingPlanId = $planId;
            $this->isEditing = true;
            $this->loadPlanData($planId);
        }
    }

    public function loadPlanData($planId)
    {
        try {
            Log::info("========== LOADING PLAN DATA ==========");
            Log::info("Loading data for plan ID: {$planId}");

            $plan = SubscriptionPlan::with(['limit', 'features', 'usageRates', 'usageRates.usageType'])
                ->findOrFail($planId);

            // Log basic plan data
            Log::info("Basic Plan Data from DB:", $plan->toArray());

            // Load basic info
            $basicFields = [
                'plan_code',
                'plan_name',
                'description',
                'plan_type',
                'duration',
                'base_price',
                'discounted_price',
                'addon_eligible',
                'is_combo',
                'plan_status',
                'max_doctors',
                'max_staff',
                'device_limit',
                'trial_period',
                'auto_renewal',
                'currency'
            ];

            $this->planData['basic'] = $plan->only($basicFields);
            Log::info("Stored Basic Data:", $this->planData['basic']);

            // Load limits
            if ($plan->limit) {
                $limitFields = [
                    'included_doctors',
                    'included_staff',
                    'storage_limit_gb',
                    'teleconsultation_minutes',
                    'sms_limit',
                    'whatsapp_limit',
                    'additional_doctor_price',
                    'additional_staff_price'
                ];
                $this->planData['limits'] = $plan->limit->only($limitFields);
                Log::info("Stored Limit Data:", $this->planData['limits']);
            } else {
                $this->planData['limits'] = [];
                Log::info("No limit data found for plan");
            }

            // Load features
            $this->planData['features'] = $plan->features->pluck('feature_id')->toArray();
            Log::info("Stored Features:", [
                'count' => count($this->planData['features']),
                'feature_ids' => $this->planData['features']
            ]);

            // Load usage rates
            $this->planData['rates'] = $plan->usageRates->map(function ($rate) {
                return [
                    'usage_type_id' => $rate->usage_type_id,
                    'rate_per_unit' => $rate->rate_per_unit,
                    'free_units' => $rate->free_units,
                    'overage_rate' => $rate->overage_rate,
                    'usage_type_name' => $rate->usageType->name ?? 'Unknown'
                ];
            })->toArray();

            //is_combo
            if ($plan->is_combo) {
                $this->planData['is_combo'] = ['plan' => [] , 'addon' => []];

                $plan->comboItems->each(function ($combo) {
                    $itemType = $combo->item_type;
                    $itemId = $combo->item_id;

                    if ($itemType === 'plan') {
                        // For plan type, store as a single integer (not in array)
                        $this->planData['is_combo'][$itemType] = $itemId;
                    } else {
                        // For other types, group items in an array
                        if (!isset($this->planData['is_combo'][$itemType])) {
                            $this->planData['is_combo'][$itemType] = [];
                        }
                        $this->planData['is_combo'][$itemType][] = $itemId;
                    }
                });
            }

            Log::info("Stored Usage Rates:", [
                'count' => count($this->planData['rates']),
                'rates' => $this->planData['rates']
            ]);

            Log::info("========== PLAN DATA LOADED ==========");
        } catch (\Exception $e) {
            Log::error("Error loading plan data: " . $e->getMessage());
            notyf()->error('Failed to load plan data');
        }
    }

    public function goToStep($step)
    {
        if ($this->isEditing || $step <= $this->highestStep) {
            $this->currentStep = $step;
        }
    }

    public function updatePlanData($step, $data)
    {
        $this->planData[$step] = $data;
        if (!$this->isEditing && $this->highestStep < $this->currentStep + 1) {
            $this->highestStep = $this->currentStep + 1;
        }
    }
    public function savePlan()
    {
        $fullData = array_merge(
            $this->planData['basic'],
            $this->planData['limits'],
            [
                'selected_features' => $this->planData['features'],
                'usage_rates' => $this->planData['rates'],
                'combo_items' => $this->planData['is_combo'],
            ]
        );

        if ($this->isEditing) {
            $this->updateExistingPlan($fullData);
        } else {
            $this->createNewPlan($fullData);
        }
    }


    protected function createNewPlan($data)
    {
        try {
            DB::beginTransaction(); // Start transaction for atomic operations

            Log::info("========== CREATING NEW PLAN ==========");
            Log::info("Full data received for creation:", $data);

            // 1. Create the base plan
            $planData = [
                'plan_code' => $data['plan_code'],
                'plan_name' => $data['plan_name'],
                'description' => $data['description'] ?? null,
                'plan_type' => $data['plan_type'],
                'duration' => $data['duration'],
                'base_price' => $data['base_price'],
                'discounted_price' => $data['discounted_price'] ?? null,
                'addon_eligible' => $data['addon_eligible'] ?? true,
                'is_combo' => $data['is_combo'] ?? false,
                'plan_status' => $data['plan_status'] ?? 'active',
                'max_doctors' => $data['max_doctors'] ?? 1,
                'max_staff' => $data['max_staff'] ?? 1,
                'device_limit' => $data['device_limit'] ?? 1,
                'trial_period' => $data['trial_period'] ?? 0,
                'auto_renewal' => $data['auto_renewal'] ?? true,
                'currency' => $data['currency'] ?? 'INR',
            ];

            Log::info("Plan data to be created:", $planData);
            $plan = SubscriptionPlan::create($planData);
            Log::info("Plan created with ID: {$plan->id}");

            // 2. Create plan limits
            $limitData = [
                'plan_id' => $plan->id,
                'included_doctors' => $data['included_doctors'] ?? 0,
                'included_staff' => $data['included_staff'] ?? 0,
                'storage_limit_gb' => $data['storage_limit_gb'] ?? 0,
                'teleconsultation_minutes' => $data['teleconsultation_minutes'] ?? 0,
                'sms_limit' => $data['sms_limit'] ?? 0,
                'whatsapp_limit' => $data['whatsapp_limit'] ?? 0,
                'additional_doctor_price' => $data['additional_doctor_price'] ?? 0,
                'additional_staff_price' => $data['additional_staff_price'] ?? 0,
            ];

            Log::info("Creating plan limits:", $limitData);
            PlanLimit::create($limitData);

            // 3. Attach features if any
            if (!empty($data['selected_features'])) {
                Log::info("Adding plan features:", [
                    'count' => count($data['selected_features']),
                    'feature_ids' => $data['selected_features']
                ]);

                foreach ($data['selected_features'] as $featureId) {
                    PlanFeature::create([
                        'plan_id' => $plan->id,
                        'feature_id' => $featureId,
                        'included' => true
                    ]);
                }
            }

            // 4. Add usage rates if any
            if (!empty($data['usage_rates'])) {
                Log::info("Adding usage rates:", [
                    'count' => count($data['usage_rates']),
                    'rates' => $data['usage_rates']
                ]);

                $usageRates = array_map(function ($rate) use ($plan, $data) {
                    return [
                        'plan_id' => $plan->id,
                        'usage_type_id' => $rate['usage_type_id'],
                        'rate_per_unit' => $rate['rate_per_unit'],
                        'free_units' => $rate['free_units'],
                        'overage_rate' => $rate['overage_rate'],
                        'currency' => $data['currency'] ?? 'INR',
                        'effective_date' => now(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }, $data['usage_rates']);

                UsageRate::insert($usageRates); // Bulk insert for better performance
            }

            // 5. Handle combo plan items
            if (!empty($data['combo_items']) && $data['is_combo']) {
                Log::info("Creating combo plan items", $data['combo_items'] ?? []);

                // Add main plan if specified
                if (!empty($data['combo_items']['plan'])) {
                    $plan->comboItems()->create([
                        'item_type' => 'plan',
                        'item_id' => $data['combo_items']['plan'],
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }

                // Add addons if specified
                if (!empty($data['combo_items']['addon'])) {
                    $addonItems = array_map(function ($addonId) use ($plan) {
                        return [
                            'combo_plan_id' => $plan->id,
                            'item_type' => 'addon',
                            'item_id' => $addonId,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ];
                    }, $data['combo_items']['addon']);

                    $plan->comboItems()->insert($addonItems); // Bulk insert
                }
            }

            DB::commit(); // Commit all changes

            Log::info("========== PLAN CREATED SUCCESSFULLY ==========");
            notyf()->success('Plan created successfully!');
            return redirect()->route('admin.subscription-plans.list');
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on error
            Log::error("Error creating plan: " . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Failed to create plan: ' . $e->getMessage());
            return back()->withInput();
        }
    }

    protected function updateExistingPlan($data)
    {
        try {
            DB::beginTransaction(); // Start transaction for atomic operations

            Log::info("========== UPDATING PLAN ==========");
            Log::info("Full data received for update:", $data);

            $plan = SubscriptionPlan::findOrFail($this->editingPlanId);
            Log::info("Current plan data:", $plan->toArray());

            $updateData = [
                'plan_code' => $data['plan_code'],
                'plan_name' => $data['plan_name'],
                'description' => $data['description'] ?? null,
                'plan_type' => $data['plan_type'],
                'duration' => $data['duration'],
                'base_price' => $data['base_price'],
                'discounted_price' => $data['discounted_price'] ?? null,
                'addon_eligible' => $data['addon_eligible'] ?? true,
                'is_combo' => $data['is_combo'] ?? false,
                'plan_status' => $data['plan_status'] ?? 'active',
                'trial_period' => $data['trial_period'] ?? 0,
                'auto_renewal' => $data['auto_renewal'] ?? true,
                'currency' => $data['currency'] ?? 'INR',
                'max_doctors' => $this->planData['basic']['max_doctors'] ?? 1,
                'max_staff' => $this->planData['basic']['max_staff'] ?? 1,
                'device_limit' => $this->planData['basic']['device_limit'] ?? 1,
            ];

            Log::info("Updating plan with data:", $updateData);
            $plan->update($updateData);
            Log::info("Plan updated successfully");

            // Update plan limits
            $limitData = [
                'included_doctors' => $data['included_doctors'] ?? 0,
                'included_staff' => $data['included_staff'] ?? 0,
                'storage_limit_gb' => $data['storage_limit_gb'] ?? 0,
                'teleconsultation_minutes' => $data['teleconsultation_minutes'] ?? 0,
                'sms_limit' => $data['sms_limit'] ?? 0,
                'whatsapp_limit' => $data['whatsapp_limit'] ?? 0,
                'additional_doctor_price' => $data['additional_doctor_price'] ?? 0,
                'additional_staff_price' => $data['additional_staff_price'] ?? 0,
            ];

            Log::info("Updating plan limits:", $limitData);
            $plan->limit()->updateOrCreate(['plan_id' => $plan->id], $limitData);

            Log::info("Deleting existing plan features");
            $plan->features()->delete();

            if (!empty($data['selected_features'])) {
                Log::info("Adding new features:", [
                    'count' => count($data['selected_features']),
                    'feature_ids' => $data['selected_features']
                ]);

                foreach ($data['selected_features'] as $featureId) {
                    PlanFeature::create([
                        'plan_id' => $plan->id,
                        'feature_id' => $featureId,
                        'included' => true
                    ]);
                }
            }

            // Update usage rates
            Log::info("Deleting existing usage rates");
            $plan->usageRates()->delete();

            if (!empty($data['usage_rates'])) {
                Log::info("Adding new usage rates:", [
                    'count' => count($data['usage_rates']),
                    'rates' => $data['usage_rates']
                ]);

                $usageRates = array_map(function ($rate) use ($plan, $data) {
                    return [
                        'plan_id' => $plan->id,
                        'usage_type_id' => $rate['usage_type_id'],
                        'rate_per_unit' => $rate['rate_per_unit'],
                        'free_units' => $rate['free_units'],
                        'overage_rate' => $rate['overage_rate'],
                        'currency' => $data['currency'] ?? 'INR',
                        'effective_date' => now(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }, $data['usage_rates']);

                UsageRate::insert($usageRates); // Bulk insert
            }

            // Handle combo plan items
            Log::info("Updating combo plan items");
            $plan->comboItems()->delete(); // Remove all existing combo items

            if (!empty($data['combo_items']) && $data['is_combo']) {
                Log::info("Creating new combo plan items", $data['combo_items'] ?? []);

                // Add main plan if specified
                if (!empty($data['combo_items']['plan'])) {
                    $plan->comboItems()->create([
                        'item_type' => 'plan',
                        'item_id' => $data['combo_items']['plan'],
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }

                // Add addons if specified
                if (!empty($data['combo_items']['addon'])) {
                    $addonItems = array_map(function ($addonId) use ($plan) {
                        return [
                            'combo_plan_id' => $plan->id,
                            'item_type' => 'addon',
                            'item_id' => $addonId,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ];
                    }, $data['combo_items']['addon']);

                    $plan->comboItems()->insert($addonItems); // Bulk insert
                }
            }

            DB::commit(); // Commit all changes

            Log::info("========== PLAN UPDATED SUCCESSFULLY ==========");
            notyf()->success('Plan updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on error
            Log::error("Error updating plan: " . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString()
            ]);
            notyf()->error('Failed to update plan: ' . $e->getMessage());
            return back()->withInput();
        }
    }

    public function render()
    {
        return view('livewire.admin.subscription-plan-wizard.wizard');
    }
}