<?php

namespace App\Livewire\Admin\Staff;

use Livewire\Component;
use App\Models\User;
use App\Models\Clinic;
use App\Models\Staff;
use Livewire\WithPagination;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

class StaffList extends Component
{
    use WithPagination;

    public $searchTerm = '';
    public $sortField = 'staff_name';
    public $sortDirection = 'asc';
    public $statusFilter = 'all';
    public $tempStatusFilter = 'all';
    public $perPage = 10;


    protected $queryString = [
        'searchTerm' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],

    ];

    public function mount()
    {
        $this->tempStatusFilter = $this->statusFilter;
    }

    public function updatingSearchTerm()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->statusFilter = $this->tempStatusFilter;
        $this->resetPage();
    }

    public function resetFilters()
    {
        $this->tempStatusFilter = 'all';
        $this->applyFilters();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function toggleStatus($staffId)
    {
        Gate::authorize('edit staff');

        try {
            $staff = Staff::findOrFail($staffId);
            $newStatus = $staff->status === 'active' ? 'inactive' : 'active';
            $staff->update(['status' => $newStatus]);

            notyf()->success('Staff status updated successfully.');
        } catch (\Exception $e) {
            notyf()->error('Failed to update staff status.');
        }
    }

    public function getAvatarColorClass($name)
    {
        $colors = ['bg-primary', 'bg-success', 'bg-info', 'bg-warning', 'bg-danger', 'bg-secondary'];
        $hash = crc32($name);
        return $colors[abs($hash) % count($colors)];
    }

    public function getInitials($name)
    {
        if (empty($name)) {
            return 'NA';
        }

        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
        } else {
            return strtoupper(substr($name, 0, 2));
        }
    }
    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $sortableFields = [
            'staff_name' => 'users.name',
            'staff_email' => 'users.email',
            'staff_status' => 'staff.status',
            'staff_created_at' => 'staff.created_at',
        ];
        $sortColumn = $sortableFields[$this->sortField] ?? 'users.name';

        $staffQuery = Staff::query()
            ->join('users', 'staff.user_id', '=', 'users.id')
            ->when($this->searchTerm, function ($query) use ($searchTerm) {
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('users.name', 'like', $searchTerm)
                        ->orWhere('users.email', 'like', $searchTerm)
                        ->orWhere('staff.phone', 'like', $searchTerm);
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('staff.status', $this->statusFilter);
            });

        $staff = $staffQuery
            ->select([
                'staff.id as staff_id',
                'users.id as user_id',
                'users.name as staff_name',
                'users.email as staff_email',
                'staff.phone as staff_phone',
                'staff.status as staff_status',
                'staff.profile_image',
                'staff.created_at as staff_created_at',
            ])
            ->orderBy($sortColumn, $this->sortDirection)
            ->paginate($this->perPage);
                
        return view('livewire.admin.staff.staff-list', [
            'staff' => $staff,
            'totalStaff' => Staff::count()
        ]);
    }
}