<?php

namespace App\Livewire\Admin\Profile;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\User;
use App\Models\UserProfile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ProfileComponent extends Component
{
    use WithFileUploads;

    public $name;
    public $email;
    public $phone_number;

    public $first_name;
    public $last_name;
    public $alternate_phone;
    public $gender;
    public $date_of_birth;
    public $timezone;

    public $profile_picture;
    public $currentProfilePicture;
    public $newProfilePicturePreview;
    public $password;
    public $password_confirmation;
    public $user;
    protected $listeners = ['dateChanged'];

    public function dateChanged($value)
    {
        $this->date_of_birth = $value;
        // Optional: Add validation or logging here
        logger()->info('Date of birth updated to: ' . $value);
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($this->user->id)
            ],
            'phone_number' => 'nullable|string|max:20',
            'first_name' => 'nullable|string|max:255',
            'last_name' => 'nullable|string|max:255',
            'alternate_phone' => 'nullable|string|max:20',
            'gender' => 'nullable|string',
            'date_of_birth' => 'nullable|date_format:d-m-Y',
            'timezone' => 'nullable|string',
            'profile_picture' => 'nullable|image|max:2048',
            'password' => 'nullable|min:8|confirmed',
        ];
    }

    public function mount()
    {
        $this->user = Auth::user();

        $this->name = $this->user->name;
        $this->email = $this->user->email;
        $this->phone_number = $this->user->phone_number;

        if ($this->user->profile) {
            $profile = $this->user->profile;
            $this->first_name = $profile->first_name;
            $this->last_name = $profile->last_name;
            $this->alternate_phone = $profile->alternate_phone;
            $this->gender = $profile->gender;
            $this->date_of_birth = $profile->date_of_birth ? $profile->date_of_birth->format('d-m-Y') : null;
            $this->timezone = $profile->timezone ?? null;
            $this->currentProfilePicture = $profile->profile_picture;
        }
    }

    public function updatedProfilePicture()
    {
        $this->newProfilePicturePreview = $this->profile_picture->temporaryUrl();
    }

    public function removeProfilePicture()
    {
        $this->profile_picture = null;
        $this->newProfilePicturePreview = null;
    }

    public function removeExistingProfilePicture()
    {
        if ($this->currentProfilePicture && Storage::disk('public')->exists($this->currentProfilePicture)) {
            Storage::disk('public')->delete($this->currentProfilePicture);
        }
        $this->currentProfilePicture = null;
        $this->user->profile()->update(['profile_picture' => null]);
    }

    public function save()
    {
        $this->validate();

        $this->user->update([
            'name' => $this->name,
            'email' => $this->email,
            'phone_number' => $this->phone_number,
        ]);

        if (!empty($this->password)) {
            $this->user->update(['password' => bcrypt($this->password)]);
        }

        $profilePicturePath = $this->currentProfilePicture;
     
        if ($this->profile_picture) {
            if ($this->currentProfilePicture && Storage::disk('public')->exists($this->currentProfilePicture)) {
                Storage::disk('public')->delete($this->currentProfilePicture);
            }

            $profilePicturePath = $this->profile_picture->store('profile-pictures', 'public');

            $this->currentProfilePicture = $profilePicturePath;
            $this->profile_picture = null;
            $this->newProfilePicturePreview = null;
        }

        $profileData = [
            'full_name' => $this->name,
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
            'alternate_phone' => $this->alternate_phone,
            'gender' => $this->gender,
            'date_of_birth' => $this->date_of_birth,
            'profile_picture' => $profilePicturePath,
        ];

        if ($this->timezone ?? null) {
            $profileData['timezone'] = $this->timezone;
        }

        $this->user->profile()->updateOrCreate(
            ['user_id' => $this->user->id],
            $profileData
        );

        $this->mount();

        notyf()->success('Profile updated successfully.');
    }

    public function render()
    {
        return view('livewire.admin.profile.profile-component');
    }
}