<?php

namespace App\Livewire\Admin\Clinics;

use App\Models\Clinic;
use App\Models\User;
use App\Models\StaffClinic;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\On;
use Livewire\WithPagination;

class StaffTable extends Component
{
    use WithPagination;
    public Clinic $clinic;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $isLoaded = false;
    public $shouldLoad = false;
    public $searchTerm = '';
    public $statusFilter = 'all';
    public $approvalStatusFilter = 'all';
    public $perPage = 10;
    public $deleteId = null;
    protected $listeners = [
        'confirm-delete-staff' => 'confirmDeleteStaff',
    ];
    public function confirmDeleteStaff($staffId, $clinicId)
    {
        if (!$staffId || !$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'Invalid staff or clinic.');
            return;
        }

        $this->deleteStaff($staffId, $clinicId);
        $this->dispatch('hide-delete-modal');
    }


    public function updatedSearchTerm()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->approvalStatusFilter = 'all';
        $this->searchTerm = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetAprovelStatusFilter()
    {
        $this->approvalStatusFilter = 'all';
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function mount(Clinic $clinic, $loadImmediately = false)
    {
        $this->clinic = $clinic;
        $this->shouldLoad = $loadImmediately;

        if ($this->shouldLoad) {
            $this->loadData();
        }
    }

    #[On('tabSwitched')]
    public function handleTabSwitch($tab)
    {
        if ($tab === 'staff' && !$this->isLoaded) {
            $this->loadData();
        }
    }

    public function loadData()
    {
        if (!$this->isLoaded) {
            $this->isLoaded = true;
        }
    }

    public function sortBy($field)
    {
        if (!$this->isLoaded) {
            $this->loadData();
        }

        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function toggleStaffStatus($id, $clinicId, $doctorId)
    {
        $user = User::findOrFail($id);
        $association = StaffClinic::where('staff_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->where('doctor_id', $doctorId)
            ->firstOrFail();

        $association->status = $association->status === 'active' ? 'inactive' : 'active';
        $association->save();

        notyf()->success('Staff Clinic status updated successfully.');
    }

    public function toggleStaffApprovalStatus($id, $clinicId, $doctorId, $status)
    {
        $user = User::findOrFail($id);
        $association = StaffClinic::where('staff_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->where('doctor_id', $doctorId)
            ->firstOrFail();

        $association->approval_status = $status;

        if ($status === 'rejected' || $status === 'pending') {
            $association->status = 'inactive';
        } elseif ($status === 'approved') {
            $association->status = 'active';
        }

        $association->save();

        notyf()->success('Staff Clinic Approval Status updated successfully.');
    }

    public function deleteStaff($id, $clinicId)
    {
        Gate::authorize('delete staff');

        if (auth()->id() == $id) {
            $this->dispatch('alert', type: 'error', message: 'You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);
        if (!$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'No clinic selected.');
            return;
        }
        $association = StaffClinic::where('staff_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->first();
        if ($association) {
            $association->delete();
            $this->deleteId = null;
        }
        notyf()->success('Staff member removed from clinic successfully.');
    }

    public function render()
    {
        $query = $this->clinic->staffAssociations()
            ->with(['staff', 'doctor'])
            ->join('users', 'staff_clinics.staff_id', '=', 'users.id')
            ->leftJoin('users as doctors', 'staff_clinics.doctor_id', '=', 'doctors.id'); // Add join for doctors

        if ($this->searchTerm) {
            $query->where(function ($subQuery) {
                $subQuery->where('users.name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('users.email', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('doctors.name', 'like', '%' . $this->searchTerm . '%'); // Add doctor name search
            });
        }

        if ($this->statusFilter !== 'all') {
            $query->where('staff_clinics.status', $this->statusFilter);
        }

        if ($this->approvalStatusFilter !== 'all') {
            $query->where('staff_clinics.approval_status', $this->approvalStatusFilter);
        }

        switch ($this->sortField) {
            case 'name':
                $query->orderBy('users.name', $this->sortDirection);
                break;
            case 'email':
                $query->orderBy('users.email', $this->sortDirection);
                break;
            case 'status':
                $query->orderBy('staff_clinics.status', $this->sortDirection);
                break;
            case 'approval_status':
                $query->orderBy('staff_clinics.approval_status', $this->sortDirection);
                break;
            default:
                $query->orderBy('users.name', 'asc');
                break;
        }

        $query->select('staff_clinics.*');

        $staffAssociations = $query->paginate($this->perPage);

        return view('livewire.admin.clinics.staff-table', [
            'staffAssociations' => $staffAssociations,
        ]);
    }
}