<?php

namespace App\Livewire\Admin\Clinics;

use Livewire\Component;
use App\Models\Clinic;
use App\Models\CustomerSubscription;
use Illuminate\Support\Str;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Gate;

class ClinicsList extends Component
{
    use WithPagination;

    public $search = '';
    public $perPage = 10;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $statusFilter = 'all';
    public $cityFilter = 'all';   
    public $countryFilter = 'all';
    public $subscriptionFilter = 'all';
    public $deleteId = null;

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'name'],
        'sortDirection' => ['except' => 'asc'],
        'statusFilter' => ['except' => 'all'],
        'cityFilter' => ['except' => 'all'],
        'countryFilter' => ['except' => 'all'],
        'subscriptionFilter' => ['except' => 'all']
    ];

    public function resetSubscriptionFilter()
    {
        $this->resetPage();
    }

    public function mount()
    {
        Gate::authorize('list clinics');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->search = '';
        $this->cityFilter = 'all';
        $this->countryFilter = 'all';
        $this->subscriptionFilter = 'all';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetCityFilter()
    {
        $this->resetPage();
    }

    public function resetCountryFilter()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function toggleStatus($id)
    {
        // Check if user has permission to edit clinics
        Gate::authorize('edit clinics');

        $clinic = Clinic::findOrFail($id);
        $newStatus = $clinic->status === 'active' ? 'inactive' : 'active';
        $clinic->update(['status' => $newStatus]);
        notyf()->success('Clinic status updated successfully.');
    }

    public function delete($id = null)
    {   
        $clinicId = $id ?? $this->deleteId;
        try {
            // Check if user has permission to delete clinics
            Gate::authorize('delete clinics');

            $clinic = Clinic::withCount([
                'staffMembers',
                'doctors',
                'patientAssociations',
                'staffAssociations',
                'doctorAssociations'
            ])->findOrFail($clinicId);

            $clinicName = $clinic->name;

            // Check for existing relationships
            $messages = [];

            if ($clinic->staff_members_count > 0) {
                $messages[] = "{$clinic->staff_members_count} staff member(s)";
            }

            if ($clinic->doctors_count > 0) {
                $messages[] = "{$clinic->doctors_count} doctor(s)";
            }

            if ($clinic->patient_associations_count > 0) {
                $messages[] = "{$clinic->patient_associations_count} patient association(s)";
            }

            if ($clinic->subscription_id) {
                $messages[] = "an active subscription";
            }

            if (!empty($messages)) {
                $messageList = implode(', ', $messages);
                throw new \Exception("Cannot delete clinic '{$clinicName}': It has {$messageList}. Please remove these first.");
            }

            \DB::transaction(function () use ($clinic) {
                // Clean up any remaining associations
                $clinic->staffAssociations()->delete();
                $clinic->doctorAssociations()->delete();

                $clinic->delete();
            });

            notyf()->success("Clinic '{$clinicName}' deleted successfully.");
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            notyf()->error("Clinic not found. It may have already been deleted.");
        } catch (\Illuminate\Auth\Access\AuthorizationException $e) {
            notyf()->error("You don't have permission to delete clinics.");
        } catch (\Illuminate\Database\QueryException $e) {
            if ($e->errorInfo[1] == 1451) { // Integrity constraint violation
                notyf()->error("Cannot delete clinic: It's currently in use by other records.");
            } else {
                notyf()->error("Database error while deleting clinic.");
            }
            \Log::error($e->getMessage());
        } catch (\Exception $e) {
            notyf()->error($e->getMessage());
            \Log::error($e->getMessage());
        }
    }

    public function render()
    {
        $clinics = Clinic::with('subscription.plan')
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                        ->orWhere('email', 'like', '%' . $this->search . '%')
                        ->orWhere('city', 'like', '%' . $this->search . '%')
                        ->orWhere('country', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('status', $this->statusFilter);
            })
            ->when($this->cityFilter !== 'all' && $this->cityFilter, function ($query) {
                $query->where('city', $this->cityFilter);
            })
            ->when($this->countryFilter !== 'all' && $this->countryFilter, function ($query) {
                $query->where('country', $this->countryFilter);
            })
            ->when($this->subscriptionFilter !== 'all', function ($query) {
                if ($this->subscriptionFilter === 'none') {
                    $query->whereNull('subscription_id');
                } else {
                    $query->whereHas('subscription', function ($q) {
                        $q->where('plan_id', $this->subscriptionFilter);
                    });
                }
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $cities = Clinic::select('city')
            ->whereNotNull('city')
            ->distinct()
            ->pluck('city');

        $countries = Clinic::select('country')
            ->whereNotNull('country')
            ->distinct()
            ->pluck('country');

        $subscriptionPlans = CustomerSubscription::with('plan')
            ->select('plan_id')
            ->whereNotNull('plan_id')
            ->distinct()
            ->get()
            ->pluck('plan.plan_name', 'plan_id')
            ->toArray();

        return view('livewire.admin.clinics.clinics-list', [
            'clinics' => $clinics,
            'cities' => $cities,
            'countries' => $countries,
            'subscriptionPlans' => $subscriptionPlans,
            'canEdit' => Gate::allows('edit clinics'),
            'canDelete' => Gate::allows('delete clinics'),
            'canCreate' => Gate::allows('create clinics')
        ]);
    }
}
