<?php

namespace App\Livewire\Admin\Auth;

use App\Mail\ForgotPasswordMail;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use App\Models\ActiveDevice;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\ValidationException;

class LoginForm extends Component
{
    public string $email = '';
    public string $password = '';
    public string $role = '';

    public function mount($role)
    {
        $this->role = $role;
    }

    protected function getDeviceRole(): string
    {
        return match ($this->role) {
            'superadmin' => 'superadmin',
            'superadminstaff' => 'staff',
            default => $this->role,
        };
    }

    protected function generateDeviceId($user): string
    {
        return hash('sha256', implode('|', [
            $user->id,
            request()->ip(),
            microtime(),
            bin2hex(random_bytes(16))
        ]));
    }

    public function login()
    {
        foreach (['web', 'superadmin', 'doctor', 'staff', 'patient'] as $guard) {
            if (Auth::guard($guard)->check()) {
                Auth::guard($guard)->logout();
                Session::flush();
                Session::regenerate();
                break;
            }
        }
        
        $this->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        Log::info('Login attempt', [
            'email' => $this->email,
            'requested_role' => $this->role
        ]);

        if (!Auth::guard('superadmin')->attempt(['email' => $this->email, 'password' => $this->password])) {
            Log::warning('Authentication failed', ['email' => $this->email]);
            throw ValidationException::withMessages(['email' => 'Invalid credentials']);
        }

        $user = Auth::guard('superadmin')->user();
        Log::info('User authenticated', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_roles' => $user->getRoleNames()->toArray(),
            'is_superadmin' => $user->is_superadmin,
            'requested_role' => $this->role
        ]);

        if ($this->role === 'superadmin' && !$user->is_superadmin) {
            Log::warning('Superadmin access denied', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'is_superadmin' => $user->is_superadmin
            ]);
            Auth::guard('superadmin')->logout();
            throw ValidationException::withMessages(['email' => 'Superadmin access denied']);
        }

        if ($this->role === 'superadminstaff' && !$user->hasRole('superadminstaff')) {
            Log::warning('Staff access denied', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'user_roles' => $user->getRoleNames()->toArray(),
                'requested_role' => $this->role
            ]);
            Auth::guard('superadmin')->logout();
            throw ValidationException::withMessages(['email' => 'Staff access denied']);
        }

        $deviceId = $this->generateDeviceId($user);
        $deviceRole = $this->getDeviceRole();
        $sessionId = Session::getId();
        $deviceName = $this->getDeviceName(request()->userAgent());

        Session::put('device_id', $deviceId);
        Session::put('current_role', $deviceRole);
        Session::put('session_id', $sessionId);

        Log::info('Setting active device', [
            'user_id' => $user->id,
            'device_role' => $deviceRole,
            'session_role' => Session::get('current_role'),
            'device_name' => $deviceName
        ]);

        ActiveDevice::updateOrCreate(
            [
                'user_id' => $user->id,
                'device_id' => $deviceId,
                'role_type' => $deviceRole,
            ],
            [
                'platform' => 'web',
                'device_name' => $deviceName,
                'ip_address' => request()->ip(),
                'login_method' => 'email',
                'is_active' => true,
                'login_at' => now(),
                'last_active_at' => now(),
                'session_id' => $sessionId,
            ]
        );        
        return redirect()->intended($this->getDashboardRoute());
    }

    protected function getDeviceName($userAgent): string
    {
        if (strpos($userAgent, 'Windows') !== false)
            return 'Windows Device';
        if (strpos($userAgent, 'Mac') !== false)
            return 'Mac Device';
        if (strpos($userAgent, 'Linux') !== false)
            return 'Linux Device';
        if (strpos($userAgent, 'iPhone') !== false)
            return 'iPhone';
        if (strpos($userAgent, 'Android') !== false)
            return 'Android Device';
        return 'Unknown Device';
    }

    protected function getDashboardRoute(): string
    {
        return match ($this->role) {
            'superadmin', 'superadminstaff' => route('admin.dashboard'),
            default => route('home')
        };
    }

    public function render()
    {
        return view('livewire.admin.auth.login-form');
    }
}