<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\DoctorClinicService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class CheckDoctorClinic
{
    protected $clinicService;

    public function __construct(DoctorClinicService $clinicService)
    {
        $this->clinicService = $clinicService;
    }

    public function handle(Request $request, Closure $next)
    {
        $user = $request->user();

        Log::info('CheckDoctorClinic Middleware: User accessed.', [
            'user_id' => $user->id,
            'roles' => $user->getRoleNames()
        ]);

        if ($user->hasRole('doctorstaff')) {
            Log::info('CheckDoctorClinic Middleware: Skipping clinic check for doctorstaff.', [
                'user_id' => $user->id
            ]);
            return $next($request);
        }

        if (!$this->clinicService->hasClinics($user)) {
            Log::warning('CheckDoctorClinic Middleware: User has no clinics.', [
                'user_id' => $user->id
            ]);

            Auth::guard('doctor')->logout();
            return redirect()->route('doctor.profile.setup')
                ->with('message', 'Please complete your clinic associations first');
        }

        Log::info('CheckDoctorClinic Middleware: Clinic check passed.', [
            'user_id' => $user->id
        ]);

        return $next($request);
    }
}
