<?php

namespace App\Http\Controllers\Patient\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\DoctorPrescription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\DoctorEMRDocument;

class PatientPrescriptionController extends Controller
{
    // Add Patient Prescriptions
    public function patientPrescriptions(Request $request)
    {
        $request->validate([
            // 'doctor_id'      => 'required|exists:users,id',
            // 'patient_id'     => 'required|exists:users,id',
            // 'appointment_id' => 'required|exists:appointments,id',
             'doctor_id'      => 'nullable|exists:users,id',        // ✅ nullable now
            'patient_id'     => 'required|exists:users,id',
            'appointment_id' => 'nullable|exists:appointments,id', // ✅ nullable now
            'diagnosis'      => 'required|string',
            'symptoms'       => 'nullable|string',
            'medications'    => 'nullable|string', // string type for form-data
            'follow_up'      => 'nullable|string',
            'notes'          => 'nullable|string',
        ]);

        $medications = $request->medications ? json_decode($request->medications, true) : [];

        $prescription = DoctorPrescription::create([
            // 'doctor_id'      => $request->doctor_id,
            // 'patient_id'     => $request->patient_id,
            // 'appointment_id' => $request->appointment_id,
            'doctor_id'      => $request->doctor_id ?: null,      // set null if 0 or empty
            'patient_id'     => $request->patient_id,
            'appointment_id' => $request->appointment_id ?: null, // set null if 0 or empty
            'type'           => 'patient', // ✅ Type explicitly set
            'diagnosis'      => $request->diagnosis,
            'symptoms'       => $request->symptoms,
            'medications'    => $medications,
            'follow_up'      => $request->follow_up,
            'notes'          => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Prescription saved successfully',
            //'data'    => $prescription
        ], 200);
    }

    // Fetch Doctor Prescriptions
    public function getPatientPrescriptions(Request $request)
    {
        // Validation
        $request->validate([
            'patient_id'      => 'required|exists:users,id',
            'appointment_id'  => 'required|exists:appointments,id',
        ]);

        // Prescriptions written by patient
        $patientPrescriptions = DoctorPrescription::where('patient_id', $request->patient_id)
                                    //->where('appointment_id', $request->appointment_id)
                                    ->where('type', 'patient')  // ✅ Only patient type
                                    ->get();

        // Prescriptions written by doctor
        $doctorPrescriptions = DoctorPrescription::where('patient_id', $request->patient_id)
                                    ->where('appointment_id', $request->appointment_id)
                                    ->where('type', 'doctor')  // ✅ Only doctor type
                                    ->get();

        // Convert medications JSON string to array if needed
        $patientPrescriptions->transform(function($item) {
            if(is_string($item->medications)) {
                $item->medications = json_decode($item->medications, true);
            }
            return $item;
        });

        $doctorPrescriptions->transform(function($item) {
            if(is_string($item->medications)) {
                $item->medications = json_decode($item->medications, true);
            }
            return $item;
        });

        return response()->json([
            'success' => true,
            'message' => 'Prescriptions fetched successfully',
            'data' => [
                'by_patient' => $patientPrescriptions,
                'by_doctor'  => $doctorPrescriptions,
            ]
        ], 200);
    }

    // Update Doctor Prescriptions
    public function updatePatientPrescriptions(Request $request)
    {
        $request->validate([
            'id'             => 'required|exists:doctor_prescriptions,id', // Prescription ID
            'patient_id'      => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
            'diagnosis'      => 'required|string',
            'symptoms'       => 'nullable|string',
            'medications'    => 'nullable|string', // form-data string
            'follow_up'      => 'nullable|string',
            'notes'          => 'nullable|string',
        ]);

        $prescription = DoctorPrescription::where('id', $request->id)
                            ->where('patient_id', $request->patient_id)
                            ->where('appointment_id', $request->appointment_id)
                            ->first();

        if (!$prescription) {
            return response()->json([
                'success' => false,
                'message' => 'Prescription not found for given doctor and appointment',
            ], 404);
        }

        $medications = $request->medications ? json_decode($request->medications, true) : [];

        $prescription->update([
            'diagnosis'   => $request->diagnosis,
            'symptoms'    => $request->symptoms,
            'medications' => $medications,
            'follow_up'   => $request->follow_up,
            'notes'       => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Prescription updated successfully',
            'data'    => $prescription
        ], 200);
    }

    // EMR Upload Document
    public function patientUploadEMR(Request $request)
    {
        $request->validate([
            'doctor_id'      => 'required|exists:users,id',
            'patient_id'     => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
            'document_type'  => 'required|string',
            'document'       => 'required|file|max:10240',
        ]);

        $file = $request->file('document');
        $filename = time().'_'.$file->getClientOriginalName();

        // File save in storage/app/public/emr
        //$path = $file->storeAs('emr', $filename, 'public');

        // Folder path inside public/storage
        $destinationPath = public_path('storage/emr');

        // Create folder if not exists
        if (!file_exists($destinationPath)) {
            mkdir($destinationPath, 0777, true);
        }

        // Move uploaded file to public/storage/emr
        $file->move($destinationPath, $filename);
        
        $emr = \App\Models\DoctorEMRDocument::create([
            'doctor_id'      => $request->doctor_id,
            'patient_id'     => $request->patient_id,
            'appointment_id' => $request->appointment_id,
            'document_type'  => $request->document_type,
            //'document_path'  => $path,
            'document_path'  => 'emr/' . $filename,  // relative path
        ]);

        // Return full public URL
        //$document_url = url('storage/' . $path);

        // Full public URL for response
        $document_url = url('storage/' . $emr->document_path);

        // return response()->json([
        //     'success' => true,
        //     'message' => 'Document uploaded successfully',
        //     'data'    => $emr
        // ], 201);

        return response()->json([
            'success'      => true,
            'message'      => 'Document uploaded successfully',
            'emr_id'       => $emr->id,
            'doctor_id'    => (string) $emr->doctor_id,
            'patient_id'   => (string) $emr->patient_id,
            'appointment_id'=> (string) $emr->appointment_id,
            'document_type'=> $emr->document_type,
            'document_path'=> $document_url,
        ], 200);
    }

    // EMR Fetch Document
    public function getpatientUploadEMR(Request $request)
    {
        $request->validate([
            'patient_id'      => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
        ]);

        $documents = DoctorEMRDocument::where('patient_id', $request->patient_id)
                        ->where('appointment_id', $request->appointment_id)
                        ->get();

        // Add full document URL
        $documents->transform(function($doc) {
            $doc->document_url = url('storage/' . $doc->document_path);
            return $doc;
        });

        return response()->json([
            'success' => true,
            'message' => 'EMR documents fetched successfully',
            'data'    => $documents
        ], 200);
    }

}
