<?php

namespace App\Http\Controllers\Patient\Api\V1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

use Carbon\Carbon;

use App\Models\ActiveDevice;
use App\Models\Appointment;
use App\Models\Clinic;
use App\Models\Doctor;
use App\Models\DoctorAvailability;
use App\Models\DoctorClinic;
use App\Models\DoctorEducation;
use App\Models\DoctorExperience;
use App\Models\DoctorProfile;
use App\Models\Patient;
use App\Models\PatientClinic;
use App\Models\User; // Assuming Patient is stored in users table with 'patient' role

class ClinicApiController extends Controller
{
    // Existing show() method
    public function show($id)
    {
        // $clinic = Clinic::findOrFail($id);
        // return response()->json([
        //     'status' => 'success',
        //     'message' => 'Clinic details fetched successfully',
        //     'data' => $clinic
        // ]);

        $clinic = Clinic::findOrFail($id);

        // ✅ Customized response
        return response()->json([
            'status' => 'success',
            'clinic_id' => $clinic->id
        ]);
    }

    // ✅ New method for QR code redirect
    public function qrRedirect($id)
    {
        // QR scan pe direct API route pe redirect
        $url = route('api.patient.v1.details', ['id' => $id]);
        return redirect($url);
    }

    // ✅ New method to add clinic for patient
    public function addClinic(Request $request)
    {
        // ✅ Validation
        $validator = Validator::make($request->all(), [
            'patient_id' => 'required|exists:users,id',
            'clinic_id'  => 'required|exists:clinics,id',
            'notes'      => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $patientId = $request->patient_id;

        // ✅ Check if already added
        $exists = PatientClinic::where('patient_id', $patientId)
                    ->where('clinic_id', $request->clinic_id)
                    ->first();

        if ($exists) {
            return response()->json([
                'status' => 'error',
                'message' => 'Clinic already added for this patient.'
            ], 409);
        }

        // ✅ Clinic ka doctor_id nikal lo
        $clinic = Clinic::find($request->clinic_id);
        $doctorId = $clinic ? $clinic->owner_id : null; // maan rahe hain `owner_id` doctor hai

        // ✅ Save new record
        $patientClinic = PatientClinic::create([
            'patient_id'      => $patientId,
            'clinic_id'       => $request->clinic_id,
            'doctor_id'       => $doctorId,
            'added_by'        => "1",
            'status'          => 'active',
            'approval_status' => 'pending',
            'notes'           => $request->notes ?? null,
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Clinic added successfully.',
            'data'    => [
                'id'              => $patientClinic->id,
                'patient_id'      => $patientClinic->patient_id,
                'clinic_id'       => $patientClinic->clinic_id,
                'doctor_id'       => $patientClinic->doctor_id,
                'status'          => $patientClinic->status,
                'approval_status' => $patientClinic->approval_status,
                'notes'           => $patientClinic->notes,
            ]
        ]);
    }

    // ✅ New method to get available clinics for patient
    public function getClinics(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'patient_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $patientId = $request->patient_id;

        // ✅ Patient clinics ke sath clinic details join
        $clinics = PatientClinic::with('clinic')
            ->where('patient_id', $patientId)
            ->get();
            
        if ($clinics->isEmpty()) {
            return response()->json([
                'status' => 'success',
                'message' => 'No clinics found for this patient.',
                'data' => []
            ], 200);
        }

        // ✅ Merge patient_clinics + clinics data
        $data = $clinics->map(function ($pc) {
            return [
                // patient_clinics table ka data
                //'patient_clinic_id' => $pc->id,
                'patient_id'        => $pc->patient_id,
                'clinic_id'         => $pc->clinic_id,
                'doctor_id'         => $pc->doctor_id,
                'added_by'          => $pc->added_by,
                'status'            => $pc->status,
                'approval_status'   => $pc->approval_status,
                'notes'             => $pc->notes,
                'name'            => $pc->clinic->name,
                'email'           => $pc->clinic->email,
                'address' => trim(
                        $pc->clinic->line1 . ', ' .
                        $pc->clinic->line2 . ', ' .
                        $pc->clinic->city . ', ' .
                        $pc->clinic->state . ', ' .
                        $pc->clinic->country . ' - ' .
                        $pc->clinic->pincode,
                        ', - '
                    ),
                'specialities'    => $pc->clinic->specialities,
                'contact_numbers' => $pc->clinic->contact_numbers,
                'rating'          => '0.0',
                //'image'         => $pc->clinic->image ? url('storage/'.$pc->clinic->image) : null,
                'image' => $pc->clinic->image ? url('storage/'.$pc->clinic->image) : asset('assets/img/hospital.png'),
            ];
        });

        return response()->json(
            // 'status'  => 'success',
            // 'message' => 'Clinics fetched successfully.',
            // 'data'    => $data
            $data
        );
    }

    // ✅ New method to get detailed clinic info for patient
    public function clinicDetails(Request $request)
    {
        $request->validate([
            'clinic_id'  => 'required|exists:clinics,id',
            'patient_id' => 'required|exists:users,id',
        ]);

        // ✅ Check patient-clinic relation
        $relation = PatientClinic::where('clinic_id', $request->clinic_id)
            ->where('patient_id', $request->patient_id)
            ->first();

        if (!$relation) {
            return response()->json([
                'status' => false,
                'message' => 'This clinic is not assigned to the patient.'
            ], 404);
        }

        // ✅ Fetch clinic
        $clinic = Clinic::with('doctors')->findOrFail($request->clinic_id);

        // ✅ Prepare response
        $data = [
            'clinic_id'      => $clinic->id,
            'name'    => $clinic->name,
            'email'   => $clinic->email,
            'details'   => $clinic->about,
            'address' => trim(
                $clinic->line1 . ', ' .
                $clinic->line2 . ', ' .
                $clinic->city . ', ' .
                $clinic->state . ', ' .
                $clinic->country . ' - ' .
                $clinic->pincode,
                ', - '
            ),
            'specialities' => is_string($clinic->specialities)
                ? explode(',', $clinic->specialities)
                : (is_array($clinic->specialities) ? $clinic->specialities : []),
            'contact_numbers' => is_string($clinic->contact_numbers)
                ? explode(',', $clinic->contact_numbers)
                : (is_array($clinic->contact_numbers) ? $clinic->contact_numbers : []),
            'status'  => $clinic->status,

            'rating'          => '0.0',
            'image'      => !empty($pc->clinic->image) ? url('storage/' . $pc->clinic->image) : asset('assets/img/hospital.png'),

            // ✅ Doctor list from relation
            'doctors' => $clinic->doctors->map(function ($doctor) {
                return [
                    'id'    => $doctor->id,
                    'name'  => $doctor->name,
                    'email' => $doctor->email,
                    'phone' => $doctor->phone ?? null,
                    'speciality' => $doctor->speciality ?? null,
                    
                ];
            })
        ];

        return response()->json([
            'status' => true,
            'message' => 'Clinic details fetched successfully',
            'data' => $data
        ]);
    }

    // ✅ Clinic ke doctor ki full profile
    public function doctorDetails(Request $request)
    {
        $request->validate([
            'clinic_id' => 'required|exists:clinics,id',
            'doctor_id' => 'required|exists:users,id',
        ]);

        $clinicId = $request->clinic_id;
        $doctorId = $request->doctor_id;

        // ✅ Check doctor assigned to clinic
        $doctorClinic = DoctorClinic::where('clinic_id', $clinicId)
            ->where('doctor_id', $doctorId)
            ->first();

        if (!$doctorClinic) {
            return response()->json([
                'status' => false,
                'message' => 'Doctor not found in this clinic'
            ], 404);
        }

        // ✅ Base user info
        $user = User::findOrFail($doctorId);

        // ✅ Doctor profile info
        $doctorProfile = DoctorProfile::where('user_id', $doctorId)->first();
        $doctor = Doctor::where('user_id', $doctorId)->first();

        // ✅ Education & Experience
        $education = DoctorEducation::where('doctor_profile_id', optional($doctorProfile)->id)->get();
        $experiences = DoctorExperience::where('doctor_profile_id', optional($doctorProfile)->id)->get();

        // ✅ Availability slots
        $availabilities = DoctorAvailability::where('doctor_id', $doctorId)
            ->where('clinic_id', $clinicId)
            ->get();

        // ✅ Response assemble
        $data = [    
            'doctor_id'    => $user->id,
            'name'  => $user->name,
            'email' => $user->email,          
            'phone' => $user->phone_number ?? null,
            'status'=> $user->status,
            'specializatioffn' => $doctorProfile->specialization ?? null,
            //'sub_specializations' => $doctorProfile->sub_specializations ? explode(',', $doctorProfile->sub_specializations) : [],
            //'years_of_experience' => $doctorProfile->years_of_experience ?? null,
            //'bio' => $doctorProfile->bio ?? null,
            //'license_number' => $doctorProfile->license_number ?? null,
            //'gender' => $doctorProfile->gender ?? null,
            //'languages_spoken' => $doctorProfile->languages_spoken ? explode(',', $doctorProfile->languages_spoken) : [],
            //'license_state' => $doctorProfile->license_state ?? null,
            //'license_valid_until' => $doctorProfile->license_valid_until ?? null,
            //'accepts_new_patients' => (bool) ($doctorProfile->accepts_new_patients ?? false),
            'rating' => $doctorProfile->rating ?? null,
            'reviews_count' => $doctorProfile->reviews_count ?? 0,
            'verification_status' => $doctorProfile->verification_status ?? null,
            //'certificates' => $doctorProfile->certificates ? json_decode($doctorProfile->certificates, true) : [],
            //'verified_at' => $doctorProfile->verified_at ?? null,
            'specialization' => $doctor->specialization ?? null,
            'qualifications' => $doctor->qualifications ?? null,
            'experience_years' => $doctor->experience_years ?? null,
            'license_number' => $doctor->license_number ?? null,
            'bio' => $doctor->bio ?? null,
            'profile_image' => $doctor->profile_image ? url('storage/'.$doctor->profile_image) : null,
            //'education' => $education,
            //'experiences' => $experiences,
            'availability' => $availabilities,
            //'clinic_relation' => $doctorClinic,
        ];

        return response()->json([
            'status' => true,
            'message' => 'Doctor full profile fetched successfully',
            'data' => $data
        ]);
    }

    // Get Slots
    public function getSlots(Request $request)
    {
        // Validation
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id',
            'date'      => 'required|date_format:Y-m-d',
        ]);

        $doctorId = $request->doctor_id;
        $date = $request->date;

        $dayOfWeek = Carbon::createFromFormat('Y-m-d', $date)->format('l');

        // Step 1: Get doctor's availability for that day
        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $doctorId)
            ->where('status', 1)
            ->get(['time_slot', 'day']);

        $availableSlots = [];

        foreach ($availability as $item) {
            $days = json_decode($item->day, true);

            if (is_array($days) && in_array($dayOfWeek, $days)) {
                $timeSlots = json_decode($item->time_slot, true);

                if (is_array($timeSlots)) {
                    $availableSlots = array_merge($availableSlots, $timeSlots);
                }
            }
        }

        // Step 2: Get already booked slots for that date
        $bookedSlotsRaw = Appointment::where('doctor_id', $doctorId)
            ->whereDate('appointment_date', $date)
            ->pluck('appointment_slot_time')
            ->toArray();

        $bookedSlots = [];
        foreach ($bookedSlotsRaw as $jsonString) {
            $slots = json_decode($jsonString, true);
            if (is_array($slots)) {
                $bookedSlots = array_merge($bookedSlots, $slots);
            }
        }

        // Step 3: Match available and booked slots
        $finalSlots = [];
        foreach ($availableSlots as $slot) {
            $slotFormatted = trim($slot);
            $isBooked = in_array($slotFormatted, $bookedSlots);
            $finalSlots[] = [
                'time'   => $slotFormatted,
                'status' => $isBooked ? 'booked' : 'available'
            ];
        }

        // ✅ If no slots found
        if (empty($finalSlots)) {
            return response()->json([
                'success' => false,
                'message' => 'No available time slots'
            ]);
        }

        return response()->json([
            'success' => true,
            'slots'   => $finalSlots
        ]);
    }

    // Book Appoinment 
    public function bookAppointment(Request $request)
    {
        // ✅ Validation
        $validated = $request->validate([
            'patient_id'            => 'required|exists:users,id',
            'doctor_id'             => 'required|exists:users,id',
            'specialization_name'   => 'required|string',
            'appointment_date'      => 'required|date_format:Y-m-d',
            'appointment_slot_time' => 'required|string', // e.g. "10:00 AM,10:30 AM"
            'total_price'           => 'required|numeric',
            'appointment_type'      => 'required|string|in:first_visit,follow_up',
            'appointment_status'    => 'required|string|in:cancelled,online,offline',
            'fee_type'              => 'required|string|in:in_person_fee,video_fee',
            'payment_type'          => 'required|string|in:cash,online',
            'reason'                => 'nullable|string|max:1000',
        ]);

        // ✅ Get Doctor's Clinic
        $clinicId = DB::table('doctor_clinic')
            ->where('doctor_id', $validated['doctor_id'])
            ->value('clinic_id');

        if (!$clinicId) {
            return response()->json([
                'success' => false,
                'message' => 'Clinic not found for this doctor'
            ], 404);
        }

        // ✅ Get Doctor booking_type
        $doctor = DB::table('doctors')
            ->where('user_id', $validated['doctor_id'])
            ->select('booking_type')
            ->first();

        if (!$doctor) {
            return response()->json([
                'success' => false,
                'message' => 'Doctor profile not found'
            ], 404);
        }

        // ✅ Check if slot already booked
        $slotTimes = array_map('trim', explode(',', $validated['appointment_slot_time']));

        $alreadyBooked = Appointment::where('doctor_id', $validated['doctor_id'])
            ->whereDate('appointment_date', $validated['appointment_date'])
            ->pluck('appointment_slot_time')
            ->toArray();

        $bookedSlots = [];
        foreach ($alreadyBooked as $jsonString) {
            $slots = json_decode($jsonString, true);
            if (is_array($slots)) {
                $bookedSlots = array_merge($bookedSlots, $slots);
            }
        }

        foreach ($slotTimes as $slot) {
            if (in_array($slot, $bookedSlots)) {
                return response()->json([
                    'success' => false,
                    'message' => "Slot {$slot} is already booked"
                ], 409);
            }
        }

        // ✅ Auto-generate appointment_id like APT00001
        $latest = Appointment::latest()->first();
        $number = $latest && $latest->appointment_id
            ? ((int) substr($latest->appointment_id, 3)) + 1
            : 1;
        $appointment_id = 'APT' . str_pad($number, 5, '0', STR_PAD_LEFT);

        // ✅ Set appointment_status based on booking_type
        $finalStatus = ($doctor->booking_type === 'manual')
            ? 'pending'
            : $validated['appointment_status'];

        // ✅ Save Appointment
        $appointment = Appointment::create([
            'appointment_id'        => $appointment_id,
            'patient_id'            => $validated['patient_id'],
            'doctor_id'             => $validated['doctor_id'],
            'clinic_id'             => $clinicId,
            'specialization'        => $validated['specialization_name'],
            'appointment_date'      => $validated['appointment_date'],
            'appointment_slot_time' => json_encode($slotTimes),
            'appointment_fee'       => $validated['total_price'],
            'fee_type'              => $validated['fee_type'],
            'appointment_type'      => $validated['appointment_type'],
            'appointment_status'    => $finalStatus, // ✅ controlled here
            'payment_type'          => $validated['payment_type'],
            'reason'                => $validated['reason'],
        ]);

        return response()->json([
            'success'     => true,
            'message'     => 'Appointment booked successfully!',
            'appointment' => $appointment
        ]);
    }

    // Get Appoinment 
    public function getAppointment(Request $request)
    {
        // ✅ Validate request params
        $validated = $request->validate([
            'patient_id'         => 'required|exists:users,id',
            'appointment_status' => 'nullable|string|in:cancelled,online,offline,completed'
        ]);

        $patientId = $validated['patient_id'];
        $status    = $validated['appointment_status'] ?? null;

        // ✅ Build Query with Doctor Relation
        $query = Appointment::with(['doctor:id,name,email'])
            ->where('patient_id', $patientId);

        if ($status) {
            $query->where('appointment_status', $status);
        }

        $appointments = $query->orderBy('id', 'DESC')->get();

        // ✅ Merge doctor details into each appointment
        $appointments->transform(function ($appointment) {
            $appointment->doctor_name  = $appointment->doctor->name ?? null;
            $appointment->doctor_email = $appointment->doctor->email ?? null;
            unset($appointment->doctor); // remove nested doctor object
            return $appointment;
        });

        return response()->json(
            //'success'      => true,
            //'message'      => $appointments->isEmpty() ? 'No appointments found' : 'Appointments fetched successfully',
             $appointments,
        );
    }

    // Appoinment Cancel
    public function updateAppointmentStatus(Request $request)
    {
        $validated = $request->validate([
            'appointment_id'     => 'required|integer|exists:appointments,id',
            'appointment_status' => 'required|string|in:cancelled', // sirf ye 2 allow
            'reason'             => 'nullable|string|max:255', // optional, sirf cancel pe use kare
        ]);

        $appointment = DB::table('appointments')->where('id', $validated['appointment_id'])->first();

        if (!$appointment) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Appointment not found.'
            ], 404);
        }

        // ✅ Update appointment status
        DB::table('appointments')
            ->where('id', $validated['appointment_id'])
            ->update([
                'appointment_status' => $validated['appointment_status'],
                'reason'             => $validated['reason'] ?? $appointment->reason,
                'updated_at'         => now(),
            ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Appointment status updated successfully.',
            'appointment_id'     => $validated['appointment_id'],
            'appointment_status' => $validated['appointment_status'],
            //'reason'             => $validated['reason'] ?? $appointment->reason,
        ], 200);
    }
}
