<?php

namespace App\Http\Controllers\Doctor\Api\V1;

use App\Http\Controllers\Controller;
use App\Models\DoctorPrescription;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\DoctorEMRDocument;

class DoctorPrescriptionController extends Controller
{
    // public function doctorPrescriptions(Request $request)
    // {
    //     // ✅ Validation directly in controller
    //     $validated = $request->validate([
    //         'doctor_id' => 'required|exists:users,id',
    //         'patient_id' => 'required|exists:users,id',
    //         'appointment_id' => 'required|exists:appointments,id',
    //         'diagnosis' => 'required|string',
    //         'symptoms' => 'required|string',
    //         'medications' => 'required|array|min:1',
    //         'medications.*.name' => 'required|string',
    //         'medications.*.dosage' => 'nullable|string',
    //         'medications.*.frequency' => 'nullable|string',
    //         'medications.*.duration' => 'nullable|string',
    //         'medications.*.instructions' => 'nullable|string',
    //         'follow_up' => 'nullable|string',
    //         'notes' => 'nullable|string',
    //     ]);

    //     // ✅ Create prescription
    //     $prescription = DoctorPrescription::create([
    //         'doctor_id'     => $validated['doctor_id'],
    //         'patient_id'    => $validated['patient_id'],
    //         'appointment_id'=> $validated['appointment_id'],
    //         'diagnosis'     => $validated['diagnosis'],
    //         'symptoms'      => $validated['symptoms'],
    //         'medications'   => $validated['medications'],
    //         'follow_up'     => $validated['follow_up'] ?? null,
    //         'notes'         => $validated['notes'] ?? null,
    //     ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Prescription created successfully',
    //         'data' => $prescription
    //     ], 201);
    // }

    // public function doctorPrescriptions(Request $request)
    // {
    //     // Validation (basic)
    //     $request->validate([
    //         'doctor_id'      => 'required|exists:users,id',
    //         'patient_id'     => 'required|exists:users,id',
    //         'appointment_id' => 'required|exists:appointments,id',
    //         'diagnosis'      => 'required|string',
    //         'symptoms'       => 'nullable|string',
    //         'medications'    => 'nullable|array',
    //         'follow_up'      => 'nullable|string',
    //         'notes'          => 'nullable|string',
    //     ]);

    //     // Save to DB
    //     $prescription = DoctorPrescription::create([
    //         'doctor_id'      => $request->doctor_id,
    //         'patient_id'     => $request->patient_id,
    //         'appointment_id' => $request->appointment_id,
    //         'diagnosis'      => $request->diagnosis,
    //         'symptoms'       => $request->symptoms,
    //         'medications'    => json_encode($request->medications), // Save JSON
    //         'follow_up'      => $request->follow_up,
    //         'notes'          => $request->notes,
    //     ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Prescription saved successfully',
    //         'data'    => $prescription
    //     ], 201);
    // }

    // Add Doctor Prescriptions
    public function doctorPrescriptions(Request $request)
    {
        $request->validate([
            'doctor_id'      => 'required|exists:users,id',
            'patient_id'     => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
            'diagnosis'      => 'required|string',
            'symptoms'       => 'nullable|string',
            'medications'    => 'nullable|string', // string type for form-data
            'follow_up'      => 'nullable|string',
            'notes'          => 'nullable|string',
        ]);

        $medications = $request->medications ? json_decode($request->medications, true) : [];

        $prescription = DoctorPrescription::create([
            'doctor_id'      => $request->doctor_id,
            'patient_id'     => $request->patient_id,
            'appointment_id' => $request->appointment_id,
            'type'           => 'doctor',
            'diagnosis'      => $request->diagnosis,
            'symptoms'       => $request->symptoms,
            'medications'    => $medications,
            'follow_up'      => $request->follow_up,
            'notes'          => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Prescription saved successfully',
            //'data'    => $prescription
        ], 201);
    }

    // Fetch Doctor Prescriptions
    public function getDoctorPrescriptions(Request $request)
    {
        // Validation
        $request->validate([
            'doctor_id'      => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
        ]);

        // Fetch prescriptions
        // $prescriptions = DoctorPrescription::where('doctor_id', $request->doctor_id)
        //                     ->where('appointment_id', $request->appointment_id)
        //                     ->get();

        // Fetch prescriptions for doctor only
        $prescriptions = DoctorPrescription::where('doctor_id', $request->doctor_id)
                    ->where('appointment_id', $request->appointment_id)
                    ->where('type', 'doctor')  // ✅ Sirf doctor ke prescriptions fetch honge
                    ->get();

        // Convert medications JSON string to array if needed
        $prescriptions->transform(function($item) {
            if(is_string($item->medications)) {
                $item->medications = json_decode($item->medications, true);
            }
            return $item;
        });

        return response()->json([
            'success' => true,
            'message' => 'Prescriptions fetched successfully',
            'data'    => $prescriptions
        ], 200);
    }

    // Update Doctor Prescriptions
    public function updateDoctorPrescriptions(Request $request)
    {
        $request->validate([
            'id'             => 'required|exists:doctor_prescriptions,id', // Prescription ID
            'doctor_id'      => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
            'diagnosis'      => 'required|string',
            'symptoms'       => 'nullable|string',
            'medications'    => 'nullable|string', // form-data string
            'follow_up'      => 'nullable|string',
            'notes'          => 'nullable|string',
        ]);

        $prescription = DoctorPrescription::where('id', $request->id)
                            ->where('doctor_id', $request->doctor_id)
                            ->where('appointment_id', $request->appointment_id)
                            ->first();

        if (!$prescription) {
            return response()->json([
                'success' => false,
                'message' => 'Prescription not found for given doctor and appointment',
            ], 404);
        }

        $medications = $request->medications ? json_decode($request->medications, true) : [];

        $prescription->update([
            'diagnosis'   => $request->diagnosis,
            'symptoms'    => $request->symptoms,
            'medications' => $medications,
            'follow_up'   => $request->follow_up,
            'notes'       => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Prescription updated successfully',
            'data'    => $prescription
        ], 200);
    }

    // EMR Upload Document
    public function doctorUploadEMR(Request $request)
    {
        $request->validate([
            'doctor_id'      => 'required|exists:users,id',
            'patient_id'     => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
            'document_type'  => 'required|string',
            'document'       => 'required|file|max:10240',
        ]);

        $file = $request->file('document');
        $filename = time().'_'.$file->getClientOriginalName();

        // File save in storage/app/public/emr
        //$path = $file->storeAs('emr', $filename, 'public');

        // Folder path inside public/storage
        $destinationPath = public_path('storage/emr');

        // Create folder if not exists
        if (!file_exists($destinationPath)) {
            mkdir($destinationPath, 0777, true);
        }

        // Move uploaded file to public/storage/emr
        $file->move($destinationPath, $filename);
        
        $emr = \App\Models\DoctorEMRDocument::create([
            'doctor_id'      => $request->doctor_id,
            'patient_id'     => $request->patient_id,
            'appointment_id' => $request->appointment_id,
            'document_type'  => $request->document_type,
            //'document_path'  => $path,
            'document_path'  => 'emr/' . $filename,  // relative path
        ]);

        // Return full public URL
        //$document_url = url('storage/' . $path);

        // Full public URL for response
        $document_url = url('storage/' . $emr->document_path);

        // return response()->json([
        //     'success' => true,
        //     'message' => 'Document uploaded successfully',
        //     'data'    => $emr
        // ], 201);

        return response()->json([
            'success'      => true,
            'message'      => 'Document uploaded successfully',
            'emr_id'       => $emr->id,
            'doctor_id'    => (string) $emr->doctor_id,
            'patient_id'   => (string) $emr->patient_id,
            'appointment_id'=> (string) $emr->appointment_id,
            'document_type'=> $emr->document_type,
            'document_path'=> $document_url,
        ], 200);
    }

    // EMR Fetch Document
    public function getDoctorUploadEMR(Request $request)
    {
        $request->validate([
            'doctor_id'      => 'required|exists:users,id',
            'appointment_id' => 'required|exists:appointments,id',
        ]);

        $documents = DoctorEMRDocument::where('doctor_id', $request->doctor_id)
                        ->where('appointment_id', $request->appointment_id)
                        ->get();

        // Add full document URL
        $documents->transform(function($doc) {
            $doc->document_url = url('storage/' . $doc->document_path);
            return $doc;
        });

        return response()->json([
            'success' => true,
            'message' => 'EMR documents fetched successfully',
            'data'    => $documents
        ], 200);
    }

}
