<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Doctor;
use App\Models\Patient;
use Carbon\Carbon;
use App\Models\Appointment;

use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function adminDashboard()
    {
        // $totalDoctors = User::whereHas('roles', function($query) {
        //     $query->where('name', 'doctor');
        // })->count();

        $totalDoctors = Doctor::count();
        $patients = Patient::count();

        $lastWeekCount = Doctor::where('created_at', '>=', Carbon::now()->subDays(7))
          ->count();

        $lastWeekPatients = Patient::where('created_at', '>=', Carbon::now()->subDays(7))
          ->count();

        $percentageChange = 0;
        if ($lastWeekCount > 0) {
            $percentageChange = (($totalDoctors - $lastWeekCount) / $lastWeekCount) * 100;
        }

        $trendPositiveDoctors = $percentageChange >= 0;

        $percentageChange = 0;
        if ($lastWeekPatients > 0) {
            $percentageChange = (($patients - $lastWeekPatients) / $lastWeekPatients) * 100;
        }

        $trendPositivePatients = $percentageChange >= 0;

        return view('admin.dashboard', [
            'totalDoctors' => $totalDoctors,
            'patients' => $patients,
            'percentageChangeDoctors' => round($percentageChange),
            'percentageChangePatients' => round($percentageChange),
            'trendPositiveDoctors' => $trendPositiveDoctors,
            'trendPositivePatients' => $trendPositivePatients
        ]);
    }       

    public function doctorDashboard()
    {
        $loggedInDoctorId = auth()->user()->id;

        // fetch all doctor_clinic rows for this doctor (may be multiple)
        $doctorClinicRecords = DB::table('doctor_clinic')
            ->where('doctor_id', $loggedInDoctorId)
            ->get();

        // collect clinic ids (unique)
        $clinicIds = $doctorClinicRecords->pluck('clinic_id')->unique()->filter()->values()->all();

        // fallback: if none found, try session or null
        if (empty($clinicIds)) {
            $sessionClinic = session('current_clinic_id');
            if ($sessionClinic) {
                $clinicIds = [$sessionClinic];
            }
        }

        // if still empty, set counts to zero to avoid queries with empty IN()
        if (empty($clinicIds)) {
            $totalAppointments = 0;
            $totalCancelledAppointments = 0;
        } else {
            // total appointments for these clinic(s)
            $totalAppointments = DB::table('appointments')
                ->whereIn('clinic_id', $clinicIds)
                ->count();

            // define what statuses count as "cancelled" in your app
            $cancelStatuses = [
                'cancelled',           // common
                'canceled',            // american spelling
                'cancelled_by_doctor',
                'cancelled_by_patient',
                'cancelled_by_user'
            ];

            // count cancelled appointments (use whereIn; adjust statuses as per your DB)
            $totalCancelledAppointments = DB::table('appointments')
                ->whereIn('clinic_id', $clinicIds)
                ->whereIn('appointment_status', $cancelStatuses)
                ->count();
        }

         // 2️⃣ Is clinic ke total unique patients count karna
        $totalPatients = DB::table('patient_clinics')
            ->where('clinic_id', $clinicIds)
            ->distinct('patient_id')
            ->count('patient_id');

        //$totalDoctors = Doctor::count();
        //$patients = Patient::count();
        //$totalPatients = Patient::whereBetween('created_at', [Carbon::now()->subDays(7), Carbon::now()])->count();

        // Current time + 1 hour
        $timeAfterOneHour = Carbon::now()->addHour();

        // Pehla upcoming appointment jo 1 ghante baad ka ho
        $appointment = DB::table('appointments')
            ->where('doctor_id', $loggedInDoctorId)
            ->where('appointment_date', '>=', $timeAfterOneHour->toDateString()) // date match
            ->where(function ($query) use ($timeAfterOneHour) {
                $query->where('appointment_date', '>', $timeAfterOneHour->toDateString())
                    ->orWhere('appointment_slot_time', '>=', $timeAfterOneHour->format('H:i:s'));
            })
            ->orderBy('appointment_date', 'asc')
            ->orderBy('appointment_slot_time', 'asc')
            ->first();

        if ($appointment) {
            $appointment = DB::table('appointments')
                ->join('users', 'appointments.patient_id', '=', 'users.id')
                ->select(
                    'appointments.*',
                    'users.name as patient_name',
                    'users.email as patient_email'
                )
                ->where('appointments.id', $appointment->id)
                ->first();
        }

        $appointments = Appointment::latest()->take(10)->get();

        // Last 12 months ke liye array
        $months = collect(range(1, 12))->map(function ($month) {
            return Carbon::create()->month($month)->format('M');
        });

        // Month-wise completed
        $completed = collect(range(1, 12))->map(function ($month) {
            return Appointment::whereMonth('created_at', $month)
                ->whereYear('created_at', Carbon::now()->year)
                ->where('appointment_status', 'completed')
                ->count();
        });

        // Month-wise cancelled
        $cancelled = collect(range(1, 12))->map(function ($month) {
            return Appointment::whereMonth('created_at', $month)
                ->whereYear('created_at', Carbon::now()->year)
                ->where('appointment_status', 'cancelled')
                ->count();
        });

        $totalCompleted = Appointment::where('appointment_status', 'completed')->count();
        $totalCancelled = Appointment::where('appointment_status', 'cancelled')->count();

        return view('doctor.doctor-dashboard', [
            'totalAppointments' => $totalAppointments,
            'totalCancelledAppointments' => $totalCancelledAppointments,
            'totalPatients' => $totalPatients,
            'appointment' => $appointment,
            'months'    => $months,
            'completed' => $completed,
            'cancelled' => $cancelled,
            'totalCompleted'  => $totalCompleted,
            'totalCancelled'  => $totalCancelled,
            'appointments' => $appointments
        ]);
    }
    public function doctorStaffDashboard()
    {
        return view('doctor.staff-dashboard');
    }

    public function patientDashboard()
    {
        $loggedInPatientId = auth()->user()->id;

        // ✅ Total Appointments for logged-in patient
        $totalAppointments = DB::table('appointments')
            ->where('patient_id', $loggedInPatientId)
            ->count();

        // ✅ Cancelled Appointments for this patient
        $cancelStatuses = [
            'cancelled', 'canceled', 'cancelled_by_doctor', 'cancelled_by_patient', 'cancelled_by_user'
        ];

        $totalCancelledAppointments = DB::table('appointments')
            ->where('patient_id', $loggedInPatientId)
            ->whereIn('appointment_status', $cancelStatuses)
            ->count();

        // ✅ Total Doctors this patient has visited (distinct doctor_id)
        $totalDoctors = DB::table('appointments')
            ->where('patient_id', $loggedInPatientId)
            ->distinct('doctor_id')
            ->count('doctor_id');

        // ✅ Sum of appointment fees for completed appointments (success)
        $totalTransactions = DB::table('appointments')
            ->where('patient_id', $loggedInPatientId)
            ->where('appointment_status', 'completed') // success condition
            ->sum('appointment_fee');

        // ✅ My Doctors list (get top 5 doctors with whom patient had appointments)
        $myDoctors = DB::table('appointments')
            ->join('doctors', 'appointments.doctor_id', '=', 'doctors.user_id')
            ->join('users', 'doctors.user_id', '=', 'users.id') // For doctor name
            ->select(
                'doctors.profile_image',
                'doctors.specialization',
                'users.name as doctor_name',
                DB::raw('COUNT(appointments.id) as total_bookings')
            )
            ->where('appointments.patient_id', $loggedInPatientId)
            ->groupBy('appointments.doctor_id', 'doctors.profile_image', 'doctors.specialization', 'users.name')
            ->orderByDesc('total_bookings')
            ->limit(5)
            ->get();

        // Fetch recent transactions
        $recentTransactions = DB::table('appointments')
            ->join('doctors', 'appointments.doctor_id', '=', 'doctors.user_id') // join with doctors table
            ->where('appointments.patient_id', $loggedInPatientId)
            ->select(
                'appointments.id',
                'appointments.appointment_fee',
                'appointments.appointment_status',
                'doctors.profile_image as doctor_image',
                'doctors.specialization',
                'doctors.user_id',
                DB::raw('(SELECT name FROM users WHERE id = doctors.user_id) as doctor_name')
            )
            ->orderBy('appointments.created_at', 'desc')
            ->limit(5)
            ->get();

        // ✅ Recent 5 Appointments for this patient
        $appointments = Appointment::where('patient_id', $loggedInPatientId)
            ->latest()
            ->take(5)
            ->get();

        return view('patient.patient-dashboard', [
            'totalAppointments' => $totalAppointments,
            'totalCancelledAppointments' => $totalCancelledAppointments,
            'totalDoctors' => $totalDoctors,
            'totalTransactions' => $totalTransactions,
            'recentTransactions' => $recentTransactions,
            'myDoctors' => $myDoctors,
            'appointments' => $appointments
        ]);
    }
}