<?php

namespace App\Http\Controllers\Admin\Api\V1;

use Log;
use App\Models\ActiveDevice;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\ValidationException;

class SuperAdminApiAuthController extends Controller
{

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
            'device_name' => 'required|string',
            'device_id' => 'sometimes|string',
            'fcm_token' => 'sometimes|string',
        ]);

        $this->clearExistingAuthSessions(); 

        if (!Auth::guard('superadmin')->attempt($request->only('email', 'password'))) {            
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        $user = Auth::guard('superadmin')->user();

        if (!$user->is_superadmin) {
            Auth::guard('superadmin')->logout();
            
            throw ValidationException::withMessages([
                'email' => ['Superadmin access denied.'],
            ]);
        }

        $tokenObject = $user->createToken($request->device_name);
        $token = $tokenObject->plainTextToken;       

        $this->trackDevice($user, $request, $token);

        return response()->json([
            'status' => 'success',
            'message' => 'Logged in successfully',
            'data' => [
                'user' => $this->getUserResponse($user),
                'access_token' => $token,
                'token_type' => 'Bearer',
            ]
        ]);
    }


    public function logout(Request $request)
    {
        try {
            $token = $request->user()->currentAccessToken();

            ActiveDevice::where('token_id', $token->id)
                ->update([
                    'is_active' => false,
                    'logout_at' => now(),
                    'fcm_token' => null,
                ]);

            $token->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Successfully logged out',
            ]);
        } catch (\Exception $e) {
            Log::error('Logout error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to logout',
            ], 500);
        }
    }


    public function user(Request $request)
    {
        return response()->json([
            'status' => 'success',
            'data' => $this->getUserResponse($request->user()),
        ]);
    }

    protected function clearExistingAuthSessions()
    {
        foreach (['web', 'superadmin', 'doctor', 'staff', 'patient'] as $guard) {
            if (Auth::guard($guard)->check()) {
                Auth::guard($guard)->logout();
                Session::flush();
                Session::regenerate();
                break;
            }
        }
    }


    protected function trackDevice($user, $request, $token, $accessToken = null)
    {
        $deviceId = $request->device_id ?? $this->generateDeviceId($user, $token);

        ActiveDevice::updateOrCreate(
            [
                'user_id' => $user->id,
                'device_id' => $deviceId,
                'role_type' => 'superadmin',
            ],
            [
                'platform' => 'android',
                'device_name' => $request->device_name,
                'ip_address' => $request->ip(),
                'login_method' => 'email',
                'is_active' => true,
                'login_at' => now(),
                'last_active_at' => now(),
                'token_id' => $accessToken ? $accessToken->id : null,
                'fcm_token' => $request->fcm_token,
            ]
        );
    }


    protected function generateDeviceId($user, $token): string
    {
        return hash('sha256', implode('|', [
            $user->id,
            $token,
            microtime(),
            bin2hex(random_bytes(16))
        ]));
    }

    protected function getUserResponse($user): array
    {
        return [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'roles' => $user->getRoleNames(),
            'permissions' => $user->getAllPermissions()->pluck('name'),
            'is_superadmin' => $user->is_superadmin,
            'created_at' => $user->created_at,
            'updated_at' => $user->updated_at,
        ];
    }
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $request->validate([
            'name' => 'sometimes|string|max:255',
            'phone_number' => 'sometimes|string|max:20|unique:users,phone_number,' . $user->id,
            'email' => 'sometimes|email|max:255|unique:users,email,' . $user->id,
            'password' => 'sometimes|string|min:8|confirmed',
        ]);

        try {
            $data = $request->only('name', 'phone_number', 'email');

            if ($request->filled('password')) {
                $data['password'] = Hash::make($request->password);
            }

            $user->update($data);

            return response()->json([
                'status' => 'success',
                'message' => 'Profile updated successfully',
                'data' => $this->getUserResponse($user),
            ]);
        } catch (\Exception $e) {
            Log::error('SuperAdmin profile update error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update profile',
            ], 500);
        }
    }
}