<?php

use Illuminate\Support\Facades\Auth;
use App\Services\StaffClinicService;

if (!function_exists('check_clinic_permission')) {
    /**
     * Check if the authenticated user has the given permission
     * for the current doctor-clinic session context.
     *
     * @param string $permission
     * @return bool
     */
    function check_clinic_permission(string $permission): bool
    {
        $user = Auth::user();
        $doctorId = session('current_doctor_id');
        $clinicId = session('current_clinic_id');       
        if (!$user || !$doctorId || !$clinicId) {
            return false;
        }
        try {
            $clinicService = app(StaffClinicService::class);
            return $clinicService->checkPermissionForPair($user, $doctorId, $clinicId, $permission);
        } catch (\Throwable $e) {
            logger()->warning("Permission check failed: {$e->getMessage()}");
            return false;
        }
    }
}

if (!function_exists('check_sub_doctor_clinic_permission')) {  // Remove the extra 'p' in 'ppermission'
    function check_sub_doctor_clinic_permission(string $permission): bool {
        $user = Auth::guard('doctor')->user();
        $clinicId = session('current_clinic_id');
        
        if ((!$user || !$clinicId) && $user->hasRole('sub_doctor|doctor')) {
            return false;
        }

        $pivot = $user->doctorClinics()->where('clinic_id' , $clinicId)->first()->pivot;
        
        if ($pivot->override_default_permissions) {
            $extraPermissions = json_decode(json_decode($pivot->extra_permissions , true), true) ?? [];
            return in_array($permission, $extraPermissions);
        }
        
        return false;
    }
    
}

if (!function_exists('getInitials')) {
    function getInitials($name)
    {
        if (empty($name)) {
            return 'NA';
        }

        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
        } else {
            return strtoupper(substr($name, 0, 2));
        }
    }
}

if (!function_exists('getAvatarColorClass')) {
    function getAvatarColorClass($name)
    {
        $colors = ['bg-primary', 'bg-success', 'bg-info', 'bg-warning', 'bg-danger', 'bg-secondary'];
        $hash = crc32($name);
        return $colors[abs($hash) % count($colors)];
    }
}