document.addEventListener('DOMContentLoaded', function () {
    window.deletePayload = {};

    window.addEventListener('show-shared-delete-modal', (event) => {
        const {
            title = 'Remove Confirmation',
            message = 'Are you sure you want to remove this item?',
            doctorId = null,
            staffId = null,
            patientId = null,
            clinicId = null,
            action = null
        } = event.detail;

        document.getElementById('deleteModalTitle').textContent = title;
        document.getElementById('deleteModalText').textContent = message;

        window.deletePayload = { doctorId, staffId, patientId, clinicId, action };
        $('#shared_delete_modal').modal('show');
    });

    window.handleSharedDelete = function () {
        const { action, ...data } = window.deletePayload;

        if (!action) return alert('No delete action specified.');

        Livewire.dispatch(action, data);
    };

    // Profile Date Picker Handler
    function initProfileDatePicker() {
        const $input = $('#date_of_birth');

        // Only initialize if the element exists on the page
        if ($input.length === 0) return;

        // Destroy existing picker if it exists
        if ($input.data('DateTimePicker')) {
            $input.data('DateTimePicker').destroy();
        }

        // Get initial value from Livewire component
        let initialValue = null;
        if (window.Livewire && window.Livewire.find) {
            try {
                const component = window.Livewire.find($input.closest('[wire\\:id]').attr('wire:id'));
                if (component && component.get('date_of_birth')) {
                    initialValue = component.get('date_of_birth');
                }
            } catch (e) {
                console.log('Could not get initial date value:', e);
            }
        }

        // Initialize the date picker
        $input.datetimepicker({
            format: 'DD-MM-YYYY',
            icons: {
                up: "fas fa-angle-up",
                down: "fas fa-angle-down",
                next: 'fas fa-angle-right',
                previous: 'fas fa-angle-left'
            },
            useCurrent: false,
            maxDate: moment(), // Prevent future dates for birth date
            defaultDate: initialValue ? moment(initialValue, 'DD-MM-YYYY') : null,

        }).on('dp.change', function (e) {
            const $this = $(this);
            const $livewireComponent = $this.closest('[wire\\:id]');

            if ($livewireComponent.length && window.Livewire) {
                const componentId = $livewireComponent.attr('wire:id');

                try {
                    if (e.date) {
                        const dateStr = e.date.format('DD-MM-YYYY');
                        $input.val(dateStr);

                        // Method 1: Using Livewire.find (preferred)
                        const component = window.Livewire.find(componentId);
                        if (component) {
                            component.set('date_of_birth', dateStr);
                        }
                    } else {
                        // Handle clear/empty date
                        $input.val('');
                        const component = window.Livewire.find(componentId);
                        if (component) {
                            component.set('date_of_birth', null);
                        }
                    }
                } catch (error) {
                    console.error('Error updating Livewire component:', error);

                    // Fallback: Use Livewire.emit
                    try {
                        if (e.date) {
                            window.Livewire.emit('dateChanged', e.date.format('DD-MM-YYYY'));
                        } else {
                            window.Livewire.emit('dateChanged', null);
                        }
                    } catch (emitError) {
                        console.error('Error emitting to Livewire:', emitError);
                    }
                }
            }
        });

        // Set initial value if exists
        if (initialValue) {
            $input.val(initialValue);
        }
    }

    // Alternative method using direct DOM manipulation (if Livewire methods fail)
    function updateLivewireDateOfBirth(dateValue) {
        const $input = $('#date_of_birth');
        const $livewireComponent = $input.closest('[wire\\:id]');

        if ($livewireComponent.length) {
            // Create a hidden input to trigger Livewire update
            let $hiddenInput = $livewireComponent.find('input[name="date_of_birth_hidden"]');

            if ($hiddenInput.length === 0) {
                $hiddenInput = $('<input type="hidden" name="date_of_birth_hidden" wire:model="date_of_birth">');
                $livewireComponent.append($hiddenInput);
            }

            $hiddenInput.val(dateValue).trigger('input');
        }
    }

    // Initialize date picker when DOM is ready
    $(document).ready(function () {
        initProfileDatePicker();
    });

    // Re-initialize after AJAX/Livewire updates
    $(document).on('livewire:load', function () {
        if (window.Livewire && window.Livewire.hook) {
            window.Livewire.hook('message.processed', function (message, component) {
                setTimeout(function () {
                    initProfileDatePicker();
                }, 100);
            });
        }
    });

    // For Livewire v3 compatibility
    $(document).on('livewire:navigated', function () {
        setTimeout(function () {
            initProfileDatePicker();
        }, 100);
    });

    // Generic function to handle any datetime picker in the application
    function initGenericDatePicker(selector, options = {}) {
        const defaultOptions = {
            format: 'DD-MM-YYYY',
            useCurrent: false,
            showTodayButton: true,
            showClear: true,
            showClose: true
        };

        const mergedOptions = Object.assign({}, defaultOptions, options);

        $(selector).each(function () {
            const $input = $(this);

            // Destroy existing picker
            if ($input.data('DateTimePicker')) {
                $input.data('DateTimePicker').destroy();
            }

            // Initialize picker
            $input.datetimepicker(mergedOptions);
        });
    }

    // Export functions for global use
    window.initProfileDatePicker = initProfileDatePicker;
    window.initGenericDatePicker = initGenericDatePicker;
    window.updateLivewireDateOfBirth = updateLivewireDateOfBirth;

});