<?php

namespace Database\Seeders;

use App\Models\Staff;
use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use App\Models\Patient;
use App\Models\UserProfile;
use App\Models\DoctorProfile;
use App\Models\DoctorClinic;
use App\Models\StaffClinic;
use App\Models\PatientClinic;
use App\Models\CustomerSubscription;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class AdminUserSeeder extends Seeder
{
    public function run(): void
    {
        // 1. Create Superadmin
        $superadmin = User::create([
            'name' => 'Super Administrator',
            'email' => 'superadmin@predcare.in',
            'password' => Hash::make('admin@123'),
            'is_superadmin' => true,
            'email_verified_at' => now(),
            'status' => 'active'
        ]);
        $Role = Role::where(['guard_name' => 'superadmin'])->where(['name' => 'superadmin'])->first();
        $superadmin->assignRole($Role);

        // 2. Create Superadmin Staff
        $superadminStaff = User::create([
            'name' => 'Admin Staff',
            'email' => 'staff@predcare.in',
            'password' => Hash::make('admin@123'),
            'email_verified_at' => now(),
            'status' => 'active'
        ]);
        $Role = Role::where(['guard_name' => 'superadmin'])->where(['name' => 'superadminstaff'])->first();
        $superadminStaff->assignRole($Role);

        $superadminStaffnew = Staff::create([
            'user_id' => $superadminStaff->id,
            'phone'    => '-1233344434',
            'profile_image' => null,
            'position' => 'staff',
            'qualifications' => 'pass',
            'status' => 'active'
        ]);



        // 4. Create Primary Doctor
        $primaryDoctor = User::create([
            'name' => 'Dr. John Doe',
            'email' => 'doctor@predcare.in',
            'password' => Hash::make('admin@123'),
            'email_verified_at' => now(),
            'status' => 'active',
        ]);
        $Role = Role::where(['guard_name' => 'doctor'])->where(['name' => 'primary_doctor'])->first();
        $primaryDoctor->assignRole($Role);

        // 3. Create Clinic
        $clinic = Clinic::create([
            'name' => 'PredCare Main Clinic',
            'email' => 'info@predcare.in',
            'status' => 'active',
            'line1' => '123 Medical Street',
            'city' => 'Mumbai',
            'state' => 'Maharashtra',
            'country' => 'India',
            'pincode' => '400001',
            'specialities' => null,
            'contact_numbers' => null,
            'owner_id' => $primaryDoctor->id,
        ]);

        //Subscription Plan
        $subscription = CustomerSubscription::create([
            'subscription_code' => 'SUB-' . strtoupper(uniqid()),
            'user_id' => $primaryDoctor->id,
            'plan_id' => 1,
            'clinic_id' => $clinic->id,
            'start_date' => now(),
            'end_date' => now()->addMonths(1),
            'renewal_date' => now()->addMonths(1),
            'status' => 'active',
            'auto_renew' => true,
            'billing_cycle' => 'monthly',
        ]);

        $clinic->update(['subscription_id' => $subscription->id]);
        $clinic->save();

        $prDocDoc = Doctor::create([
            'user_id' => $primaryDoctor->id,
            'specialization' => 'Cardiology',
            'qualifications' => 'MBBS',
            'experience_years' => 7,
            'license_number' => 'DOC561932',
            'bio' => 'I am Primary Doctor and Senior Cardiologist with 10+ years experience.',
        ]);

        // Create Doctor Profile
        $doctorProfile = DoctorProfile::create([
            'user_id' => $primaryDoctor->id,
            'specialization' => $prDocDoc->specialization,
            'sub_specializations' => ['Interventional Cardiology', 'Echocardiography'],
            'years_of_experience' => $prDocDoc->experience_years,
            'bio' => $prDocDoc->bio,
            'languages_spoken' => ['English', 'Hindi', 'Marathi'],
            'license_number' => $prDocDoc->license_number,
            'license_state' => 'Maharashtra',
            'license_valid_until' => now()->addYears(5),
            'accepts_new_patients' => true,
            'verification_status' => 'verified',
            'verified_at' => now(),
            'verified_by' => $superadmin->id,
        ]);

        // Associate Doctor with Clinic as Primary
        $this->doctorClinicAssociation(
            $primaryDoctor->id,
            $clinic->id,
            $superadmin->id,
            true,
        );

        // 4. Create Primary Doctor 1
        $primaryDoctor1 = User::create([
            'name' => 'Dr. Shoen Doe',
            'email' => 'doctor1@predcare.in',
            'password' => Hash::make('admin@123'),
            'email_verified_at' => now(),
            'status' => 'active',
        ]);
        $Role = Role::where(['guard_name' => 'doctor'])->where(['name' => 'primary_doctor'])->first();
        $primaryDoctor1->assignRole($Role);

        // Create second clinic for demonstration
        $secondClinic = Clinic::create([
            'name' => 'PredCare Secondary Clinic',
            'email' => 'branch@predcare.in',
            'status' => 'active',
            'line1' => '456 Health Avenue',
            'city' => 'Mumbai',
            'state' => 'Maharashtra',
            'country' => 'India',
            'pincode' => '400002',
            'specialities' => null,
            'contact_numbers' => null,
            'owner_id' => $primaryDoctor1->id,
        ]);

        //Subscription Plan
        $secondSubscription = CustomerSubscription::create([
            'subscription_code' => 'SUB-' . strtoupper(uniqid()),
            'user_id' => $primaryDoctor1->id,
            'plan_id' => 1,
            'clinic_id' => $secondClinic->id,
            'start_date' => now(),
            'end_date' => now()->addMonths(1),
            'renewal_date' => now()->addMonths(1),
            'status' => 'active',
            'auto_renew' => true,
            'billing_cycle' => 'monthly',
        ]);

        $secondClinic->update(['subscription_id' => $secondSubscription->id]);
        $secondClinic->save();

        $prDocDoc1 = Doctor::create([
            'user_id' => $primaryDoctor1->id,
            'specialization' => 'Radiology',
            'qualifications' => 'MBBS',
            'experience_years' => 7,
            'license_number' => 'DOC561234',
            'bio' => 'I am Primary Doctor1.',
        ]);

        // Create Doctor Profile
        $doctorProfile = DoctorProfile::create([
            'user_id' => $primaryDoctor1->id,
            'specialization' => $prDocDoc1->specialization,
            'sub_specializations' => ['Interventional Cardiology', 'Echocardiography'],
            'years_of_experience' => $prDocDoc1->experience_years,
            'bio' => $prDocDoc1->bio,
            'languages_spoken' => ['English', 'Hindi', 'Marathi'],
            'license_number' => $prDocDoc1->license_number,
            'license_state' => 'Maharashtra',
            'license_valid_until' => now()->addYears(5),
            'accepts_new_patients' => true,
            'verification_status' => 'verified',
            'verified_at' => now(),
            'verified_by' => $superadmin->id,
        ]);

        // Associate Doctor with Clinic as Primary
        $this->doctorClinicAssociation(
            $primaryDoctor1->id,
            $secondClinic->id,
            $superadmin->id,
            true,
        );

        // 5. Create Sub Doctor
        $subDoctor = User::create([
            'name' => 'Dr. Jane Smith',
            'email' => 'juniordoctor@predcare.in',
            'password' => Hash::make('admin@123'),
            'email_verified_at' => now(),
            'status' => 'active',
        ]);
        $Role = Role::where(['guard_name' => 'doctor'])->where(['name' => 'sub_doctor'])->first();
        $subDoctor->assignRole($Role);

        $subDocDoc = Doctor::create([
            'user_id' => $subDoctor->id,
            'specialization' => 'Radiology',
            'qualifications' => 'MBBS',
            'experience_years' => 3,
            'license_number' => 'DOC561332',
            'bio' => 'I am Sub Doctor Under Dr.John Doe .',
        ]);

        DoctorProfile::create([
            'user_id' => $subDoctor->id,
            'specialization' => $subDocDoc->specialization,
            'sub_specializations' => ['Family Medicine'],
            'years_of_experience' => $subDocDoc->experience_years,
            'bio' => $subDocDoc->bio,
            'languages_spoken' => ['English', 'Hindi'],
            'license_number' => $subDocDoc->license_number,
            'license_state' => 'Maharashtra',
            'license_valid_until' => now()->addYears(3),
            'accepts_new_patients' => true,
            'verification_status' => 'verified',
            'verified_at' => now(),
            'verified_by' => $superadmin->id,
        ]);

        $this->doctorClinicAssociation(
            $subDoctor->id,
            $clinic->id,
            $superadmin->id,
            false,
        );

        // 6. Create Doctor Staff - Clinic Staff (can work with multiple doctors and clinics)
        $doctorStaff = User::create([
            'name' => 'Clinic Staff',
            'email' => 'clinicstaff@predcare.in',
            'password' => Hash::make('admin@123'),
            'phone_number' => '-9876521637',
            'email_verified_at' => now(),
            'status' => 'active',
        ]);
        $Role = Role::where(['guard_name' => 'doctor'])->where(['name' => 'doctorstaff'])->first();
        $doctorStaff->assignRole($Role);
        $doctorStaffnew = Staff::create([
            'user_id' => $doctorStaff->id,
            'phone'    => '-9876521637',
            'profile_image' => null,
            'position' => 'staff',
            'qualifications' => 'pass',
            'status' => 'active'
        ]);
        $staffProfile = UserProfile::create([
            'user_id' => $doctorStaff->id,
            'full_name' => $doctorStaff->name,
            'first_name' => 'Clinic',
            'last_name' => 'Staff',
            'alternate_phone' => '-6542134375',
            'gender' => 'male',
            'date_of_birth' => '1998-07-11',
            'profile_picture' => null,
        ]);

        // Associate staff with clinics
        $this->staffClinicAssociation(
            $doctorStaff->id,
            $clinic->id,
            $subDoctor->id,
            $superadmin->id,
            [
                'access staff dashboard',
                'list patients',
                'view patient profile',
                #'list appointments',
                #'create appointments',
                #'edit appointments',
                #'view schedules'
            ]
        );

        $this->staffClinicAssociation(
            $doctorStaff->id,
            $secondClinic->id,
            $primaryDoctor1->id,
            $superadmin->id,
            [
                'access staff dashboard',
                'list patients',
                'view patient profile',
                #'list appointments',
                #'create appointments',
                #'edit appointments',
                #'view schedules'
            ]
        );

        $this->staffClinicAssociation(
            $doctorStaff->id,
            $clinic->id,
            $primaryDoctor->id,
            $superadmin->id,
            [
                'access staff dashboard',
                'list patients',
                'view patient profile',
                #'list appointments',
                #'create appointments',
                #'edit appointments',
                #'view schedules'
            ]
        );

        $patient = User::create([
            'name' => 'Patient User',
            'email' => 'patient@predcare.in',
            'password' => Hash::make('admin@123'),
            'email_verified_at' => now(),
            'status' => 'active',
        ]);
        $Role = Role::where(['guard_name' => 'patient'])->where(['name' => 'patient'])->first();
        $patient->assignRole($Role);

        $patientPatient = Patient::create([
            'user_id' => $patient->id,
            'address' => '123 Main Street, Apartment 4B',
            'street_address' => '123 Main Street',
            'city' => 'New York',
            'state' => 'NY',
            'postal_code' => '10001',
            'country' => 'United States',
            'phone' => '-1 (555) 123-4567',
            'date_of_birth' => '1985-06-15',  // YYYY-MM-DD format
            'gender' => 'male',
            'medical_history' => 'Patient has a history of mild asthma. No known drug allergies. Last physical exam completed 2023-01-10.',
            'profile_image' => null, // or default image path if available
            'created_at' => now(),
            'updated_at' => now()
        ]);

        // Associate patient with clinics
        $this->patientClinicAssociation($patient->id, $clinic->id, $primaryDoctor->id, $superadmin->id);
        $this->patientClinicAssociation($patient->id, $clinic->id, $subDoctor->id, $superadmin->id);
        $this->patientClinicAssociation($patient->id, $secondClinic->id, $primaryDoctor1->id, $superadmin->id);

        $this->command->info('Admin users and clinic seeded successfully!');
        $this->command->table(
            ['Type', 'Email', 'Password', 'Clinic Association'],
            [
                ['Superadmin', 'superadmin@predcare.in', 'admin@123', 'N/A'],
                ['Superadmin Staff', 'staff@predcare.in', 'admin@123', 'N/A'],
                ['Primary Doctor', 'doctor@predcare.in', 'admin@123', $clinic->name . ' (Primary)'],
                ['Primary Doctor1', 'doctor1@predcare.in', 'admin@123', $secondClinic->name . ' (Primary)'],
                ['Sub Doctor', 'juniordoctor@predcare.in', 'admin@123', $clinic->name],
                ['Doctor Staff', 'clinicstaff@predcare.in', 'admin@123', $clinic->name . ' & ' . $secondClinic->name],
                ['Patient', 'patient@predcare.in', 'admin@123', 'N/A']
            ]
        );
    }

    public function doctorClinicAssociation($doctorId, $clinicId, $addedBy, $isPrimary = false, $fees = [])
    {
        $defaultFees = [
            'in_person_fee' => 800.00,
            'video_fee' => 600.00,
            'home_visit_fee' => 1500.00,
            'in_person_duration' => 20,
            'video_duration' => 20,
            'home_visit_duration' => 45,
            'offers_in_person' => true,
            'offers_video' => true,
            'offers_home_visit' => false,
        ];

        $attributes = array_merge($defaultFees, $fees);

        return DoctorClinic::create([
            'doctor_id' => $doctorId,
            'clinic_id' => $clinicId,
            'added_by' => $addedBy,
            'is_primary' => $isPrimary,
            'status' => 'active',
            'approval_status' => 'approved',
            'approved_at' => now(),
            'in_person_fee' => $attributes['in_person_fee'],
            'video_fee' => $attributes['video_fee'],
            'home_visit_fee' => $attributes['home_visit_fee'],
            'in_person_duration' => $attributes['in_person_duration'],
            'video_duration' => $attributes['video_duration'],
            'home_visit_duration' => $attributes['home_visit_duration'],
            'offers_in_person' => $attributes['offers_in_person'],
            'offers_video' => $attributes['offers_video'],
            'offers_home_visit' => $attributes['offers_home_visit'],
        ]);
    }

    public function staffClinicAssociation($staffId, $clinicId, $doctorId, $addedBy, $permissions = [])
    {
        return StaffClinic::create([
            'staff_id' => $staffId,
            'clinic_id' => $clinicId,
            'doctor_id' => $doctorId,
            'added_by' => $addedBy,
            'status' => 'active',
            'approval_status' => 'approved',
            'approved_at' => now(),
            'permissions' => json_encode($permissions)
        ]);
    }

    public function patientClinicAssociation($patientId, $clinicId, $doctorId, $addedBy)
    {
        return PatientClinic::create([
            'patient_id' => $patientId,
            'clinic_id' => $clinicId,
            'doctor_id' => $doctorId,
            'added_by' => $addedBy,
            'status' => 'active',
            'approval_status' => 'approved',
            'approved_at' => now(),
        ]);
    }
}
