<?php

namespace App\Services;

use App\Models\User;
use App\Models\DoctorClinic;
use Illuminate\Support\Facades\Log;

class DoctorClinicService
{
    public function getAvailableClinics(User $user): array
    {
        if (!$user->isDoctor()) {
            Log::info("User {$user->id} is not a doctor. Returning empty clinic list.");
            return [];
        }

        Log::info("Fetching clinics directly from DB for doctor {$user->id}");
        return DoctorClinic::with('clinic')
            ->where('doctor_id', $user->id)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->get()
            ->map(function ($association) {
                return $this->formatClinicData($association);
            })
            ->toArray();
    }

    public function getCurrentClinic(User $user): ?array
    {
        if (!$user->isDoctor()) {
            Log::info("User {$user->id} is not a doctor. No current clinic context.");
            return null;
        }

        if ($currentClinicId = session('current_clinic_id')) {
            Log::info("Found session clinic_id: {$currentClinicId} for doctor {$user->id}");
            $association = $this->getClinicAssociation($user, $currentClinicId);
            if ($association) {
                Log::info("Returning session clinic for doctor {$user->id}");
                return $association;
            }
        }

        if ($primary = $this->getPrimaryClinic($user)) {
            Log::info("No session clinic. Found primary clinic for doctor {$user->id}");
            $this->setCurrentClinicInSession($primary['id']);
            return $primary;
        }

        if ($first = $this->getFirstAvailableClinic($user)) {
            Log::info("No session or primary clinic. Using first available clinic for doctor {$user->id}");
            $this->setCurrentClinicInSession($first['id']);
            return $first;
        }

        Log::warning("No clinic found for doctor {$user->id}");
        return null;
    }

    public function hasClinics(User $user): bool
    {
        if (!$user->isDoctor()) {
            Log::info("User {$user->id} is not a doctor. Cannot have clinics.");
            return false;
        }

        $hasClinics = DoctorClinic::where('doctor_id', $user->id)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->exists();

        Log::info("Doctor {$user->id} has clinics: " . ($hasClinics ? 'yes' : 'no'));
        return $hasClinics;
    }

    public function getPrimaryClinic(User $user): ?array
    {
        $association = DoctorClinic::with('clinic')
            ->where('doctor_id', $user->id)
            ->where('is_primary', true)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->first();

        if ($association) {
            Log::info("Primary clinic found for doctor {$user->id}: {$association->clinic_id}");
        } else {
            Log::info("No primary clinic found for doctor {$user->id}");
        }

        return $association ? $this->formatClinicData($association) : null;
    }

    public function setCurrentClinicInSession(int $clinicId): void
    {
        session(['current_clinic_id' => $clinicId]);
        Log::info("Session set: current_clinic_id = {$clinicId}");
    }

    public function clearCache(User $user): void
    {
        Log::info("Cache cleared skipped: caching disabled for doctor {$user->id}");
    }

    protected function getFirstAvailableClinic(User $user): ?array
    {
        $association = DoctorClinic::with('clinic')
            ->where('doctor_id', $user->id)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->first();

        if ($association) {
            Log::info("First available clinic for doctor {$user->id} is {$association->clinic_id}");
        } else {
            Log::info("No available clinics for doctor {$user->id}");
        }

        return $association ? $this->formatClinicData($association) : null;
    }

    public function getClinicAssociation(User $user, int $clinicId): ?array
    {
        Log::channel('permissions')->info('Loading clinic association', [
            'user_id' => $user->id,
            'clinic_id' => $clinicId,
            'user_roles' => $user->getRoleNames()
        ]);

        $association = DoctorClinic::with('clinic')
            ->where('doctor_id', $user->id)
            ->where('clinic_id', $clinicId)
            ->where('status', 'active')
            ->where('approval_status', 'approved')
            ->first();

        if ($association) {
            Log::channel('permissions')->info('Clinic association loaded', [
                'user_id' => $user->id,
                'clinic_id' => $clinicId,
                'is_primary' => $association->is_primary,
                'approval_status' => $association->approval_status
            ]);
        } else {
            Log::channel('permissions')->warning('No clinic association found', [
                'user_id' => $user->id,
                'clinic_id' => $clinicId,
                'available_clinics' => $this->getAvailableClinics($user)
            ]);
        }

        return $association ? $this->formatClinicData($association) : null;
    }

    public function getStaffAssignedClinics($staff)
    {
        return $staff->staffClinics()
            ->where('status', 'active')
            ->with('clinic')
            ->get()
            ->map(function ($staffClinic) {
                return [
                    'id' => $staffClinic->clinic->id,
                    'name' => $staffClinic->clinic->name,
                    'location' => $staffClinic->clinic->location ?? $staffClinic->clinic->address,
                    'role_type' => 'staff',
                    'is_primary' => false,
                    'doctor_id' => $staffClinic->doctor_id,
                    'permissions' => $staffClinic->permissions
                ];
            })->toArray();
    }

    public function getStaffClinicAssociation($staff, $clinicId)
    {
        return $staff->staffClinics()
            ->where('clinic_id', $clinicId)
            ->where('status', 'active')
            ->first();
    }

    protected function formatClinicData(DoctorClinic $association): array
    {
        return [
            'id' => $association->clinic_id,
            'name' => $association->clinic->name,
            'is_primary' => $association->is_primary,
            'association_id' => $association->id,
            'clinic_data' => $association->clinic,
            'role_type' => $association->is_primary ? 'primary_doctor' : 'sub_doctor',
            'permissions' => $association->getEffectivePermissions()
        ];
    }
}
