<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DoctorProfile extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'phone_number',
        'specialization',
        'sub_specializations',
        'years_of_experience',
        'bio',
        'languages_spoken',
        'license_number',
        'license_state',
        'license_valid_until',
        'accepts_new_patients',
        'rating',
        'reviews_count',
        'verification_status',
        'verified_at',
        'verified_by',
        'certificates'
    ];

    protected $casts = [
        'sub_specializations' => 'array',
        'languages_spoken' => 'array',
        'license_valid_until' => 'date',
        'verified_at' => 'datetime',
        'accepts_new_patients' => 'boolean',
        'certificates' => 'array'
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function verifier()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function educations()
    {
        return $this->hasMany(DoctorEducation::class);
    }

    public function experiences()
    {
        return $this->hasMany(DoctorExperience::class);
    }

    public function clinics()
    {
        return $this->belongsToMany(Clinic::class, 'doctor_clinic')
            ->withPivot([
                'is_primary',
                'status',
                'in_person_fee',
                'video_fee',
                'home_visit_fee',
                'in_person_duration',
                'video_duration',
                'home_visit_duration',
                'offers_in_person',
                'offers_video',
                'offers_home_visit'
            ])
            ->withTimestamps();
    }

    public function doctor()
    {
        return $this->hasOne(Doctor::class, 'user_id', 'user_id');
    }

    public function getProfileImageAttribute()
    {
        return $this->doctor?->profile_image;
    }

    protected static function booted()
    {
        static::saving(function ($doctorProfile) {
            if ($doctorProfile->isDirty('phone_number')) {
                $doctorProfile->user->update([
                    'phone_number' => $doctorProfile->phone_number
                ]);
            }
        });

        static::saved(function ($doctorProfile) {
            if ($doctorProfile->isDirty('specialization') && $doctorProfile->doctor) {
                $doctorProfile->doctor->update([
                    'specialization' => $doctorProfile->specialization
                ]);
            }
        });
    }
}
