<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Relations\Pivot;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Support\Facades\DB;

class DoctorClinic extends Pivot
{
    protected $table = 'doctor_clinic';

    protected $fillable = [
        'doctor_id',
        'clinic_id',
        'added_by',
        'subscription_member_id',
        'is_primary',
        'status',
        'approval_status',
        'in_person_fee',
        'video_fee',
        'home_visit_fee',
        'in_person_duration',
        'video_duration',
        'home_visit_duration',
        'offers_in_person',
        'offers_video',
        'offers_home_visit',
        'extra_permissions',
        'disabled_permissions',
        'override_default_permissions',
        'device_limit',
        'approved_at'
    ];

    protected $casts = [
        'is_primary' => 'boolean',
        'offers_in_person' => 'boolean',
        'offers_video' => 'boolean',
        'offers_home_visit' => 'boolean',
        'override_default_permissions' => 'boolean',
        'in_person_fee' => 'decimal:2',
        'video_fee' => 'decimal:2',
        'home_visit_fee' => 'decimal:2',
        'approved_at' => 'datetime',
        'in_person_duration' => 'integer',
        'video_duration' => 'integer',
        'home_visit_duration' => 'integer',
        'extra_permissions' => 'array',
        'disabled_permissions' => 'array'
    ];

    public function doctor()
    {
        return $this->belongsTo(User::class, 'doctor_id');
    }

    public function clinic()
    {
        return $this->belongsTo(Clinic::class, 'clinic_id');
    }

    public function addedBy()
    {
        return $this->belongsTo(User::class, 'added_by');
    }

    public function subscriptionMember()
    {
        return $this->belongsTo(DoctorSubscriptionMember::class, 'subscription_member_id');
    }

    protected function inPersonFee(): Attribute
    {
        return Attribute::make(
            get: fn($value) => $value ? (float) $value : null,
            set: fn($value) => $value ? number_format($value, 2, '.', '') : null
        );
    }

    protected function videoFee(): Attribute
    {
        return Attribute::make(
            get: fn($value) => $value ? (float) $value : null,
            set: fn($value) => $value ? number_format($value, 2, '.', '') : null
        );
    }

    protected function homeVisitFee(): Attribute
    {
        return Attribute::make(
            get: fn($value) => $value ? (float) $value : null,
            set: fn($value) => $value ? number_format($value, 2, '.', '') : null
        );
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeApproved($query)
    {
        return $query->where('approval_status', 'approved');
    }

    public function scopePrimary($query)
    {
        return $query->where('is_primary', true);
    }

    public function markAsApproved()
    {
        $this->update([
            'approval_status' => 'approved',
            'approved_at' => now()
        ]);
    }

    public function setAsPrimary()
    {
        DB::transaction(function () {
            static::where('clinic_id', $this->clinic_id)
                ->where('is_primary', true)
                ->update(['is_primary' => false]);

            $this->update(['is_primary' => true]);
        });
    }

    public function getServiceTypesAttribute()
    {
        $services = [];
        if ($this->offers_in_person) $services[] = 'In Person';
        if ($this->offers_video) $services[] = 'Video';
        if ($this->offers_home_visit) $services[] = 'Home Visit';
        return $services;
    }

    public function getEffectivePermissions()
    {
        if ($this->override_default_permissions) {
            return $this->extra_permissions ?? [];
        }

        $defaultPermissions = $this->getDefaultRolePermissions();
        $extraPermissions = $this->extra_permissions ?? [];
        $disabledPermissions = $this->disabled_permissions ?? [];

        return array_diff(
            array_unique(array_merge($defaultPermissions, $extraPermissions)),
            $disabledPermissions
        );
    }

    protected function getDefaultRolePermissions()
    {
        return $this->is_primary 
            ? config('permissions.default.primary_doctor', [])
            : config('permissions.default.sub_doctor', []);
    }
}