<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Clinic extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'name',
        'email',
        'subscription_id',
        'about',
        'line1',
        'line2',
        'city',
        'state',
        'country',
        'pincode',
        'location',
        'specialities',
        'contact_numbers',
        'status',
        'owner_id',
        'qr_code', // ✅ Added for QR code storage
    ];

    protected $casts = [
        'location' => 'array',
        'specialities' => 'array',
        'contact_numbers' => 'array'
    ];
    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function activeSubscription()
    {
        return $this->owner->activeSubscription ?? null;
    }

    public function staffMembers()
    {
        return $this->hasManyThrough(
            User::class,
            StaffClinic::class,
            'clinic_id',
            'id',
            'id',
            'staff_id'
        )->whereHas('roles', fn($q) => $q->where('name', 'doctorstaff'));
    }
    public function getSpecialitiesAttribute($value)
    {
        return $value ? json_decode($value, true) : [];
    }
    public function subscription()
    {
        return $this->belongsTo(CustomerSubscription::class, 'subscription_id');
    }

    public function doctors()
    {
        return $this->belongsToMany(User::class, 'doctor_clinic', 'clinic_id', 'doctor_id')
            ->using(DoctorClinic::class)
            ->withPivot([
                'is_primary',
                'status',
                'in_person_fee',
                'video_fee',
                'home_visit_fee',
                'added_by',
                'subscription_member_id'
            ]);
    }


    public function primaryDoctor()
    {
        return $this->hasOne(DoctorClinic::class, 'clinic_id')
            ->where('is_primary', true)
            ->with('doctor');
    }

    public function doctorAssociations()
    {
        return $this->hasMany(DoctorClinic::class, 'clinic_id');
    }
    public function staffAssociations()
    {
        return $this->hasMany(StaffClinic::class, 'clinic_id');
    }

    public function setPrimaryDoctor(User $doctor)
    {
        if (!$this->doctors()->where('users.id', $doctor->id)->exists()) {
            throw new \Exception('Doctor is not associated with this clinic');
        }

        \DB::transaction(function () use ($doctor) {
            $this->doctorAssociations()->update(['is_primary' => false]);
            $this->doctorAssociations()
                ->where('doctor_id', $doctor->id)
                ->update(['is_primary' => true]);
        });

        return $this->fresh()->primaryDoctor;
    }

    public function assignSubscription(CustomerSubscription $subscription)
    {
        if ($this->subscription_id && $this->subscription_id !== $subscription->id) {
            throw new \Exception('Clinic already has an active subscription');
        }

        $this->update(['subscription_id' => $subscription->id]);
        return $this;
    }

    public function removeSubscription()
    {
        return $this->update(['subscription_id' => null]);
    }

    public function hasActiveSubscription()
    {
        return $this->subscription_id && optional($this->subscription)->status === 'active';
    }
    public function scopeActive($query)
    {
        return $query->whereHas('subscription', function ($q) {
            $q->where('status', 'active');
        });
    }

    public function patientAssociations()
    {
        return $this->hasMany(PatientClinic::class, 'clinic_id');
    }

    public function mount(Clinic $clinic)
    {
        $clinic->load(['patientAssociations.patient', 'patientAssociations.doctor']);
        $this->clinic = $clinic;
    }
    
}