<?php

namespace App\Livewire\Patient\Auth;

use Illuminate\Support\Facades\Log;
use App\Models\User;
use Livewire\Component;
use App\Models\ActiveDevice;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\ValidationException;

class LoginForm extends Component
{
    public string $email = '';
    public string $mobile = '';
    public string $password = '';
    public string $role = '';
    public bool $remember = false;
    public bool $useMobile = false;

    public function mount($role)
    {
        $this->role = $role;
        Log::info('Login form mounted', ['role' => $this->role]);
    }

    protected function getDeviceRole(): string
    {
        return match ($this->role) {
            'patient' => 'patient',
            default => 'user',
        };
    }

    protected function generateDeviceId($user): string
    {
        return hash('sha256', implode('|', [
            $user->id,
            request()->ip(),
            microtime(),
            bin2hex(random_bytes(16))
        ]));
    }

    public function toggleLoginMethod()
    {
        $this->useMobile = !$this->useMobile;
        $this->resetValidation();
    }

    public function login()
    {
        foreach (['web', 'superadmin', 'doctor', 'staff', 'patient'] as $guard) {
            if (Auth::guard($guard)->check()) {
                Auth::guard($guard)->logout();
                Session::flush();
                Session::regenerate();
                break;
            }
        }
        if ($this->useMobile) {
            $this->validate([
                'mobile' => 'required|string|min:10',
                'password' => 'required',
            ]);
        } else {
            $this->validate([
                'email' => 'required|email',
                'password' => 'required',
            ]);
        }

        Log::info('Patient login attempt', [
            'login_method' => $this->useMobile ? 'mobile' : 'email',
            'identifier' => $this->useMobile ? $this->mobile : $this->email,
            'requested_role' => $this->role
        ]);

        $credentials = [
            'password' => $this->password,
            $this->useMobile ? 'mobile' : 'email' => $this->useMobile ? $this->mobile : $this->email
        ];

        if (!Auth::guard('patient')->attempt($credentials, $this->remember)) {
            Log::warning('Poctor authentication failed', [
                'identifier' => $this->useMobile ? $this->mobile : $this->email
            ]);
            throw ValidationException::withMessages([
                $this->useMobile ? 'mobile' : 'email' => 'Invalid credentials'
            ]);
        }

        $user = Auth::guard('patient')->user();
        Log::info('Patient user authenticated', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_mobile' => $user->mobile,
            'user_roles' => $user->roles->pluck('name')->toArray(),
            'requested_role' => $this->role
        ]);

        // Check role access based on URL
        if (!$this->validatePatientRole($user)) {
            Auth::guard('patient')->logout();
            throw ValidationException::withMessages([
                $this->useMobile ? 'mobile' : 'email' => 'Access denied'
            ]);
        }

        if ($user->hasRole(['patient']) && !$user->patient) {
            Auth::guard('patient')->logout();
            Log::info('Patient Authentication Failed User Is Not A Patient.');
            notyf()->error('Patient Authentication Failed User Is Not A Patient.');
            return;
        }


        $deviceId = $this->generateDeviceId($user);
        $deviceRole = $this->getDeviceRole(); // Now returns shorter codes
        $sessionId = Session::getId();
        $deviceName = $this->getDeviceName(request()->userAgent());

        Session::put('device_id', $deviceId);
        Session::put('current_role', $deviceRole);
        Session::put('session_id', $sessionId);

        Log::info('Setting active Patient device', [
            'user_id' => $user->id,
            'device_role' => $deviceRole,
            'session_role' => Session::get('current_role'),
            'device_name' => $deviceName
        ]);

        try {
            ActiveDevice::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'device_id' => $deviceId,
                    'role_type' => $deviceRole,
                ],
                [
                    'platform' => 'web',
                    'device_name' => $deviceName,
                    'ip_address' => request()->ip(),
                    'login_method' => $this->useMobile ? 'mobile' : 'email',
                    'is_active' => true,
                    'login_at' => now(),
                    'last_active_at' => now(),
                    'session_id' => $sessionId,
                ]
            );
        } catch (\Exception $e) {
            Log::error('Failed to update active device', [
                'error' => $e->getMessage(),
                'user_id' => $user->id,
                'device_role' => $deviceRole
            ]);
            // Continue with login even if device tracking fails
        }

        return redirect()->intended($this->getDashboardRoute());
    }
    protected function validatePatientRole(User $user)
    {
        if ($this->role === 'patient' && !$user->hasRole(['patient'])) {
            return false;
        }
        return true;
    }

    protected function getDeviceName($userAgent): string
    {
        if (strpos($userAgent, 'Windows') !== false)
            return 'Windows Device';
        if (strpos($userAgent, 'Mac') !== false)
            return 'Mac Device';
        if (strpos($userAgent, 'Linux') !== false)
            return 'Linux Device';
        if (strpos($userAgent, 'iPhone') !== false)
            return 'iPhone';
        if (strpos($userAgent, 'Android') !== false)
            return 'Android Device';
        return 'Unknown Device';
    }

    protected function getDashboardRoute(): string
    {
        return match ($this->role) {
            'patient' => route('patient.patient-dashboard'),
            default => route('home')
        };
    }

    public function render()
    {
        return view('livewire.patient.auth.login-form', [
            'isMobileLogin' => $this->useMobile
        ]);
    }
}