<?php

namespace App\Livewire\Doctor\Staff;

use Livewire\Component;
use App\Models\User;
use App\Models\Staff;
use App\Models\StaffClinic;
use Illuminate\Support\Facades\Log;
use Livewire\WithPagination;

class StaffView extends Component
{
    use WithPagination;
    public $staff;
    public $staffRecord;
    public $staffId;
    public $staffClinics = [];
    public $permissions = [];
    public $notes = [];

    public function mount($staffId = null)
    {
        try {
            $this->staffId = $staffId ?? request()->route('staffId') ?? request()->get('staffId');

            if (!$this->staffId || !is_numeric($this->staffId)) {
                throw new \Exception("Invalid staff ID provided");
            }

            $this->loadStaffDetails();
        } catch (\Exception $e) {
            Log::error('Staff details loading failed: ' . $e->getMessage());
            notyf()->error('Failed to load staff details: ' . $e->getMessage());
            return redirect()->route('doctor.staff.list');
        }
    }

    protected function loadStaffDetails()
    {
        $this->staffRecord = Staff::where('user_id', $this->staffId)->first();

        if (!$this->staffRecord) {
            throw new \Exception("Staff member not found or invalid");
        }

        $this->staff = User::with([
            'staff',
            'staffClinics' => function ($query) {
                $query->with(['clinic:id,name', 'doctor:id,name,email'])
                    ->orderBy('created_at', 'desc');
            },
            'roles'
        ])->find($this->staffId);

        if (!$this->staff) {
            throw new \Exception("Staff member not found or invalid");
        }

        $this->staffClinics = $this->staff->staffClinics->map(function ($staffClinic) {
            return [
                'id' => $staffClinic->id,
                'clinic' => $staffClinic->clinic,
                'doctor' => $staffClinic->doctor,
                'status' => $staffClinic->status,
                'approval_status' => $staffClinic->approval_status,
                'permissions' => $staffClinic->permissions,
                'notes' => $staffClinic->notes,
                'created_at' => $staffClinic->created_at,
                'approved_at' => $staffClinic->approved_at,
                'added_by_name' => $staffClinic->addedBy ? $staffClinic->addedBy->name : 'N/A'
            ];
        })->toArray();
    }

    public function addStaff()
    {
        return redirect()->route('doctor.staff.add');
    }
    public function editStaff($doctor, $clinic_id)
    {
        if ($clinic_id != session('current_clinic_id')) {
            notyf()->error('You are not allowed to edit another clinic\'s Staff Association.');
            return;
        }

        return redirect()->route('doctor.staff.edit', [
            'staff' => $this->staff,
            'doctor' => $doctor['id'],
        ]);
    }


    public function removeStaff($staff_id, $doctor_id, $clinic_id)
    {
        if ($clinic_id != session('current_clinic_id')) {
            notyf()->error('You are not allowed to remove another clinic\'s Staff Association.');
            return;
        }
        try {
            if (auth()->id() == $staff_id) {
                notyf()->error('You cannot delete your own account.');
                return;
            }

            $hasOtherClinics = StaffClinic::where('staff_id', $staff_id)->exists();
            StaffClinic::where('staff_id', $staff_id)
                ->where('clinic_id', $clinic_id)
                ->where('doctor_id', $doctor_id)
                ->delete();

            if (!$hasOtherClinics) {
                User::find($staff_id)->delete();
            }

            notyf()->success('Staff removed successfully.');
        } catch (\Exception $e) {
            notyf()->error('Failed to delete staff: ' . $e->getMessage());
        }
    }


    public function toggleStatus()
    {
        try {
            $newStatus = $this->staffRecord->status === 'active' ? 'inactive' : 'active';
            $this->staffRecord->update(['status' => $newStatus]);

            $this->loadStaffDetails();

            notyf()->success('Staff status updated successfully.');
        } catch (\Exception $e) {
            Log::error('Status toggle failed: ' . $e->getMessage());
            notyf()->error('Failed to update staff status.');
        }
    }

    public function toggleClinicStatus($staffClinicId)
    {
        try {
            $staffClinic = StaffClinic::with('clinic.subscription.plan')->findOrFail($staffClinicId);
            $newStatus = $staffClinic->status === 'active' ? 'inactive' : 'active';

            if ($newStatus === 'active') {
                $clinic = $staffClinic->clinic;
                if (!$clinic || !$clinic->subscription || !$clinic->subscription->plan) {
                    throw new \Exception('Clinic subscription information is missing or invalid.');
                }

                $maxStaff = $clinic->subscription->plan->max_staff;
                $currentActiveStaffCount = StaffClinic::where('clinic_id', $clinic->id)
                    ->where('status', 'active')
                    ->count();

                if ($currentActiveStaffCount >= $maxStaff) {
                    notyf()->error("Cannot activate. The clinic has reached its maximum active staff limit of {$maxStaff}.");
                    return;
                }
            }

            $staffClinic->update(['status' => $newStatus]);

            $this->loadStaffDetails();

            notyf()->success('Clinic assignment status updated successfully.');
        } catch (\Exception $e) {
            Log::error('Clinic status toggle failed: ' . $e->getMessage());
            notyf()->error('Failed to update clinic assignment status.');
        }
    }

    public function backToList()
    {
        return redirect()->route('doctor.staff.list');
    }

    public function render()
    {
        return view('livewire.doctor.staff.staff-view');
    }
}
