<?php

namespace App\Livewire\Doctor\Profile;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\UserProfile;
use App\Models\Staff;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class StaffProfileComponent extends Component
{
    use WithFileUploads;

    public $name;
    public $email;
    public $phone_number;
    public $phone;
    public $date_of_birth;
    public $gender;
    public $profile_image;
    public $first_name;
    public $last_name;
    public $timezone;
    public $currentProfilePicture;
    public $newProfilePicturePreview;

    public $password;
    public $password_confirmation;

    public $userProfile;
    public $staff;
    public $user;
    protected $listeners = ['dateChanged'];

    public function dateChanged($value)
    {
        $this->date_of_birth = $value;
        // Optional: Add validation or logging here
        //logger()->info('Date of birth updated to: ' . $value);
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'first_name' => 'required|string|max:55',
            'last_name' => 'required|string|max:55',
            'timezone' => 'required|string|max:20',
            'email' => [
                'required',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore(Auth::id())
            ],
            'phone_number' => 'nullable|string|max:20',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other,prefer_not_to_say',
            'profile_image' => 'nullable|image|max:2048',
            'password' => 'nullable|min:8|confirmed',
        ];
    }

    public function mount()
    {
        $user = Auth::guard('doctor')->user();
        $this->user = $user;

        $this->name = $user->name;
        $this->email = $user->email;
        $this->phone_number = $user->phone_number;

        $this->staff = Staff::where('user_id', $user->id)->first();

        if ($this->staff && $this->staff->profile_image) {
            $this->currentProfilePicture = $this->staff->profile_image;
        }

        if ($user->profile) {
            $this->userProfile = $user->profile;

            $this->first_name = optional($this->userProfile)->first_name;
            $this->last_name = optional($this->userProfile)->last_name;
            $this->timezone = optional($this->userProfile)->timezone;
            $this->phone = optional($this->userProfile)->alternate_phone;
            $this->date_of_birth = optional($this->userProfile)->date_of_birth;
            $this->gender = optional($this->userProfile)->gender;
        }
    }

    public function updatedProfileImage()
    {
        $this->newProfilePicturePreview = $this->profile_image->temporaryUrl();
    }

    public function removeProfilePicture()
    {
        $this->profile_image = null;
        $this->newProfilePicturePreview = null;
    }

    public function removeExistingProfilePicture()
    {
        if ($this->currentProfilePicture && Storage::disk('public')->exists($this->currentProfilePicture)) {
            Storage::disk('public')->delete($this->currentProfilePicture);
        } else {
            Log::warning('File not found in public disk');
        }

        $this->currentProfilePicture = null;
        
        if ($this->staff) {
            $this->staff->update(['profile_image' => null]);
        }
    }

    public function save()
    {
        $this->validate();

        $user = User::findOrFail(Auth::id());
        $user->update([
            'name' => $this->name,
            'email' => $this->email,
            'phone_number' => $this->phone_number,
        ]);

        if (!empty($this->password)) {
            $user->update(['password' => bcrypt($this->password)]);
        }

        if ($this->profile_image) {
            if ($this->currentProfilePicture && Storage::disk('public')->exists($this->currentProfilePicture)) {
                Storage::disk('public')->delete($this->currentProfilePicture);
            }

            $profilePicturePath = $this->profile_image->store('staff/profile_images', 'public');
            $this->currentProfilePicture = $profilePicturePath;

            if ($this->staff) {
                $this->staff->update(['profile_image' => $profilePicturePath]);
            } else {
                $this->staff = Staff::create([
                    'user_id' => $user->id,
                    'profile_image' => $profilePicturePath,
                    'phone' => $this->phone_number,
                    'status' => 'active'
                ]);
            }

            $this->profile_image = null;
            $this->newProfilePicturePreview = null;
        }

        $data = [
            'first_name' => $this->first_name,
            'last_name' => $this->last_name,
        ];

        if($this->date_of_birth){
            $data['date_of_birth'] = $this->date_of_birth;
        }
        if($this->gender){
            $data['gender'] = $this->gender;
        }
        if($this->phone){
            $data['alternate_phone'] = $this->phone;
        }
        if($this->timezone){
            $data['timezone'] = $this->timezone;
        }
        if($this->userProfile){
            $this->userProfile->updateOrCreate($data);
        } else {
            $data['user_id'] = $this->user->id;
            $this->userProfile = UserProfile::create($data);
            
        }

        notyf()->success('Profile updated successfully.');
    }

    public function getProfileImageUrl()
    {
        if ($this->profile_image && is_object($this->profile_image)) {
            try {
                return $this->profile_image->temporaryUrl();
            } catch (\Exception $e) {
                Log::error('Failed to get temporary URL: ' . $e->getMessage());
                return null;
            }
        }

        if ($this->currentProfilePicture) {
            if (Storage::disk('public')->exists($this->currentProfilePicture)) {
                return asset('storage/' . $this->currentProfilePicture);
            } else {
                Log::warning('Profile image file not found', [
                    'path' => $this->currentProfilePicture,
                    'full_path' => storage_path('app/public/' . $this->currentProfilePicture)
                ]);
                return null;
            }
        }

        return null;
    }

    public function render()
    {
        return view('livewire.doctor.profile.staff-profile-component');
    }
}