<?php

namespace App\Livewire\Doctor\Profile;

use Livewire\Component;
use App\Models\DoctorExperience;
use App\Models\DoctorProfile;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class DoctorExperienceComponent extends Component
{
    public $experiences = [];

    protected $rules = [
        'experiences.*.position' => 'required|string|max:255',
        'experiences.*.organization' => 'required|string|max:255',
        'experiences.*.start_date' => 'required|date',
        'experiences.*.end_date' => 'nullable|date|after:experiences.*.start_date',
    ];

    public function mount()
    {
        $doctorProfile = Auth::user()->doctorProfile;
        $this->experiences = $doctorProfile->experiences->map(function($exp) {
            return [
                'id' => $exp->id,
                'position' => $exp->position,
                'organization' => $exp->organization,
                'start_date' => $exp->start_date->format('Y-m-d'),
                'end_date' => $exp->end_date ? $exp->end_date->format('Y-m-d') : null,
                'is_current' => $exp->is_current,
            ];
        })->toArray();
        
        if (empty($this->experiences)) {
            $this->addExperience();
        }
    }

    public function addExperience()
    {
        $this->experiences[] = [
            'position' => '',
            'organization' => '',
            'start_date' => '',
            'end_date' => null,
            'is_current' => false,
        ];
    }

    public function removeExperience($index)
    {
        if (isset($this->experiences[$index]['id'])) {
            DoctorExperience::find($this->experiences[$index]['id'])->delete();
        }
        unset($this->experiences[$index]);
        $this->experiences = array_values($this->experiences);
    }

    public function save()
    {
        $this->validate();

        $doctorProfile = Auth::user()->doctorProfile;

        $doctorProfile->experiences()->delete();

        foreach ($this->experiences as $experience) {
            if (!empty($experience['position']) && !empty($experience['organization'])) {
                $data = [
                    'position' => $experience['position'],
                    'organization' => $experience['organization'],
                    'start_date' => $experience['start_date'],
                    'is_current' => empty($experience['end_date']),
                ];

                if($experience['end_date'] ?? null){
                    $data['end_date'] = $experience['end_date'];
                }
                $doctorProfile->experiences()->create($data);
            }
        }

        notyf()->success('Experience information saved successfully.');
    }

    public function render()
    {
        return view('livewire.doctor.profile.doctor-experience-component');
    }
}
