<?php

namespace App\Livewire\Admin\Staff;

use Livewire\Component;
use App\Models\User;
use App\Models\Clinic;
use App\Models\StaffClinic;
use App\Models\Staff;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Livewire\WithFileUploads;



class StaffManagement extends Component
{
    use WithFileUploads;

    public $name, $email, $password, $password_confirmation;
    public $phone_number;
    public $status = 'active';
    public $editMode = false;
    public $userIdBeingEdited = null;

    public $staffMode = 'create';
    public $existingUsers = [];
    public $selectedExistingUser;
    public $searchTerm = '';
    public $dropdownVisible = false;
    public $profileImage;
    public $currentProfileImage;

    protected function rules()
    {
        $rules = [
            'status' => 'required|in:active,inactive',
            'phone_number' => 'nullable|string|max:20',
            'profileImage' => 'nullable|image|max:2048',
        ];

        if ($this->staffMode === 'select') {
            $rules['selectedExistingUser'] = 'required|exists:users,id';
        } else {
            $rules['name'] = 'required|string|max:255';
            $rules['email'] = 'required|email|max:255|unique:users,email' . ($this->userIdBeingEdited ? ',' . $this->userIdBeingEdited : '');
            if (!$this->editMode) {
                $rules['password'] = 'required|min:6|confirmed';
            } elseif ($this->password) {
                $rules['password'] = 'required|min:6|confirmed';
            }
        }

        return $rules;
    }

    public function mount()
    {
        $this->existingUsers = [];

        $this->loadExistingUsers();

        // Only process edit if we have a numeric ID
        $routeId = request()->route('id');
        $queryEditId = request()->get('edit_id');
        $queryId = request()->get('id');

        $editId = $routeId ?? $queryEditId ?? $queryId;

        if ($editId && is_numeric($editId)) {
            $this->edit($editId);
        }
    }

    public function toggleStaffMode($mode)
    {
        $this->resetValidation();
        $this->staffMode = $mode;

        if ($mode === 'create') {
            $this->selectedExistingUser = null;
            $this->reset(['name', 'email']);
        }
    }

    public function updatedSearchTerm()
    {
        $this->dropdownVisible = !empty($this->searchTerm);
        $this->loadExistingUsers();
    }

    public function loadExistingUsers()
    {
        $query = User::query()
            ->where('id', '!=', auth()->id())
            ->orderBy('name');

        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->existingUsers = $query->take(100)->get();
    }

    public function selectUser($userId)
    {
        $this->selectedExistingUser = $userId;
        $this->dropdownVisible = false;
        $this->searchTerm = '';
    }

    public function updatedSelectedExistingUser($value)
    {
        if ($value) {
            $user = User::find($value);
            if ($user) {
                $this->name = $user->name;
                $this->email = $user->email;
                $this->phone_number = $user->phone_number ?? '';
            }
        }
    }

    public function createStaff()
    {
        $this->validate();

        try {
            DB::transaction(function () {
                if ($this->staffMode === 'select') {
                    $this->assignExistingUserAsStaff();
                } else {
                    $this->createNewStaff();
                }
            });

            notyf()->success('Staff created successfully.');
            $this->resetForm();
            return redirect()->route('admin.staff-list');
        } catch (\Exception $e) {
            Log::error('Staff creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create staff: ' . $e->getMessage());
        }
    }

    protected function assignExistingUserAsStaff()
    {
        $user = User::findOrFail($this->selectedExistingUser);

        Staff::updateOrCreate(
            ['user_id' => $user->id],
            [
                'phone' => $this->phone_number,
                'status' => $this->status,
            ]
        );

        if (!$user->hasRole('doctorstaff')) {
            $staffRole = Role::where('name', 'doctorstaff')
                ->where('guard_name', 'doctor')
                ->firstOrFail();

            $user->assignRole($staffRole);
        }

    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'email',
            'password',
            'password_confirmation',
            'phone_number',
            'status',
            'userIdBeingEdited',
            'editMode',
            'staffMode',
            'existingUsers',
            'selectedExistingUser',
            'searchTerm',
            'dropdownVisible',
            'profileImage',
            'currentProfileImage'
        ]);
        $this->resetValidation();
    }

    protected function createNewStaff()
    {
        if (User::where('email', $this->email)->exists()) {
            throw new \Exception('Email already exists. Please use a different email.');
        }

        try {
            DB::transaction(function () {
                $user = User::create([
                    'name' => $this->name,
                    'email' => $this->email,
                    'phone_number' => $this->phone_number,
                    'password' => Hash::make($this->password),
                    'status' => $this->status,
                ]);

                $staffRole = Role::where('name', 'doctorstaff')
                    ->where('guard_name', 'doctor')
                    ->firstOrFail();

                $user->assignRole($staffRole);

                $staffData = [
                    'user_id' => $user->id,
                    'phone' => $this->phone_number,
                    'status' => $this->status,
                ];

                // Handle file upload
                if ($this->profileImage) {
                    $path = $this->profileImage->store('staff/profile_images', 'public');
                    $staffData['profile_image'] = $path;
                }

                Staff::create($staffData);

            });

            //notyf()->success('Staff created successfully.');
            $this->resetForm();
            return redirect()->route('admin.staff-list');
        } catch (\Exception $e) {
            Log::error('Staff creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create staff: ' . $e->getMessage());
        }
    }

    public function updateStaff()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $this->userIdBeingEdited,
            'status' => 'required|in:active,inactive',
            'phone_number' => 'nullable|string|max:20',
            'profileImage' => 'nullable|image|max:2048',
        ];

        if ($this->password) {
            $rules['password'] = 'required|min:6|confirmed';
        }

        $this->validate($rules);

        try {
            DB::transaction(function () {
                $user = User::findOrFail($this->userIdBeingEdited);

                $updateData = [
                    'name' => $this->name,
                    'email' => $this->email,
                    'phone_number' => $this->phone_number,
                    'status' => $this->status,
                ];

                if ($this->password) {
                    $updateData['password'] = Hash::make($this->password);
                }

                $user->update($updateData);

                // Update staff record
                $staffData = [
                    'phone' => $this->phone_number,
                    'status' => $this->status,
                ];

                // Handle profile image update
                if ($this->profileImage) {
                    // Delete old image if exists
                    if ($user->staff && $user->staff->profile_image) {
                        Storage::disk('public')->delete($user->staff->profile_image);
                    }

                    $path = $this->profileImage->store('staff/profile_images', 'public');
                    $staffData['profile_image'] = $path;
                }

                // Update or create staff record
                if ($user->staff) {
                    $user->staff->update($staffData);
                } else {
                    $staffData['user_id'] = $user->id;
                    Staff::create($staffData);
                }

            });

            notyf()->success('Staff updated successfully.');
            $this->resetForm();
            return redirect()->route('admin.staff-list');
        } catch (\Exception $e) {
            Log::error('Staff update failed: ' . $e->getMessage());
            notyf()->error('Failed to update staff: ' . $e->getMessage());
        }
    }

    public function edit($id)
    {
        try {
            if (!is_numeric($id)) {
                throw new \Exception("Invalid ID format: " . $id);
            }

            $user = User::with([
                'staff',
                'staffClinics' => function ($query) {
                    $query->where('status', 'active')
                        ->latest();
                }
            ])->find($id);

            if (!$user || !$user->staff) {
                throw new \Exception("Staff not found");
            }

            $this->editMode = true;
            $this->userIdBeingEdited = $id;
            $this->name = $user->name;
            $this->email = $user->email;
            $this->phone_number = $user->phone_number;
            $this->status = $user->staff->status; 
            $this->password = '';
            $this->password_confirmation = '';
            if ($user->staff && $user->staff->profile_image) {
                $this->currentProfileImage = $user->staff->profile_image;
            }

            $this->dispatch('scroll-to-form');
        } catch (\Exception $e) {
            Log::error('Edit failed:', ['error' => $e->getMessage(), 'id' => $id]);
            notyf()->error('Failed to load staff data: ' . $e->getMessage());
        }
    }

    public function save()
    {
        if ($this->editMode) {
            $this->updateStaff();
        } else {
            $this->createStaff();
        }
    }

    public function render()
    {
        return view('livewire.admin.staff.staff-management');
    }
}