<?php

namespace App\Livewire\Admin\Plans;

use Livewire\Component;
use App\Models\UsageType;
use Illuminate\Support\Str;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Log;

class UsageTypeList extends Component
{
    use WithPagination;

    public $search = '';
    public $perPage = 5;
    public $sortField = 'type_name';
    public $sortDirection = 'asc';
    public $statusFilter = 'all';
    public $unitNameFilter = 'all';
    public $deleteId = null;

    protected $queryString = [
        'search' => ['except' => ''],
        'sortField' => ['except' => 'type_name'],
        'sortDirection' => ['except' => 'asc'],
        'statusFilter' => ['except' => 'all'],
        'unitNameFilter' => ['except' => 'all']
    ];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->unitNameFilter = 'all';
        $this->search = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetUnitNameFilter()
    {
        $this->unitNameFilter = 'all';
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function toggleStatus($id)
    {
        $usageType = UsageType::findOrFail($id);
        $usageType->update([
            'is_active' => !$usageType->is_active
        ]);

        notyf()->success('Usage Type status updated successfully');
    }

    public function delete($id)
    {
        try {
            $usageType = UsageType::findOrFail($id);
            $usageTypeName = $usageType->type_name;

            // Check for existing relationships
            $messages = [];

            if ($usageType->usageRates()->exists()) {
                $messages[] = "usage rates";
            }

            if ($usageType->usageTransactions()->exists()) {
                $messages[] = "usage transactions";
            }

            if ($usageType->freeUnitBalances()->exists()) {
                $messages[] = "free unit balances";
            }

            if (!empty($messages)) {
                $messageList = implode(', ', $messages);
                throw new \Exception("Cannot delete Usage Type: It has associated {$messageList}. Please remove these first.");
            }

            $usageType->delete();
            notyf()->success("Usage Type '{$usageTypeName}' deleted successfully");
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            notyf()->error("Usage Type not found. It may have already been deleted.");
        } catch (\Illuminate\Database\QueryException $e) {
            if ($e->errorInfo[1] == 1451) { // Integrity constraint violation
                notyf()->error("Cannot delete Usage Type: It's currently in use by other records.");
            } else {
                notyf()->error("Database error while deleting Usage Type.");
            }
            \Log::error($e->getMessage());
        } catch (\Exception $e) {
            notyf()->error($e->getMessage());
            \Log::error($e->getMessage());
        }
    }

    public function render()
    {
        $usageTypes = UsageType::query()
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('type_name', 'like', '%' . $this->search . '%')
                        ->orWhere('unit_name', 'like', '%' . $this->search . '%')
                        ->orWhere('description', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('is_active', $this->statusFilter === 'active');
            })
            ->when($this->unitNameFilter !== 'all' && !empty($this->unitNameFilter), function ($query) {
                $query->where('unit_name', $this->unitNameFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        // Get distinct unit names for filter dropdown
        $unitNames = UsageType::select('unit_name')
            ->whereNotNull('unit_name')
            ->where('unit_name', '!=', '')
            ->distinct()
            ->orderBy('unit_name')
            ->pluck('unit_name');

        return view('livewire.admin.plans.usage-type-list', [
            'usageTypes' => $usageTypes,
            'unitNames' => $unitNames
        ]);
    }
}
