<?php

namespace App\Livewire\Admin\Plans;

use App\Models\AddOn;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Str;

class AddonsList extends Component
{
    use WithPagination;

    public $search = '';
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $statusFilter = 'all';
    public $resourceTypeFilter = 'all';
    public $tempStatusFilter = 'all';
    public $tempResourceTypeFilter = 'all';
    public $deleteId = null;
    public $perPage = 10;

    protected $queryString = [
        'search' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'resourceTypeFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
    ];

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function mount()
    {
        $this->tempStatusFilter = $this->statusFilter;
        $this->tempResourceTypeFilter = $this->resourceTypeFilter;
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->statusFilter = $this->tempStatusFilter;
        $this->resourceTypeFilter = $this->tempResourceTypeFilter;
        $this->resetPage();
    }

    public function resetFilters()
    {
        $this->tempStatusFilter = 'all';
        $this->tempResourceTypeFilter = 'all';
        $this->applyFilters();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function toggleStatus($addonId)
    {
        try {
            $addon = AddOn::findOrFail($addonId);
            $addon->is_active = !$addon->is_active;
            $addon->save();
            notyf()->success('Addon status updated successfully.');
        } catch (\Exception $e) {
            notyf()->error('Failed to update addon status.');
        }
    }

    public function delete($addonId)
    {
        try {
            $addon = AddOn::findOrFail($addonId);
            $addonName = $addon->name;


            if ($addon->addonItems()->exists()) {
                $comboCount = $addon->addonItems()->count();
                throw new \Exception("This addon is currently used in {$comboCount} Combo(s). "
                    . "Please remove it from all plans before deleting.");
            }

            if ($addon->subscriptionAddOns()->exists()) {
                $clinicCount = $addon->subscriptionAddOns()->count();
                throw new \Exception("This addon is currently used in {$clinicCount} Clinic. "
                    . "Please remove it from all plans before deleting.");
            }

            $addon->delete();
            notyf()->success("Addon '{$addonName}' was deleted successfully.");
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            notyf()->error("Addon not found. It may have already been deleted.");
        } catch (\Illuminate\Database\QueryException $e) {
            if ($e->errorInfo[1] == 1451) {
                notyf()->error("Cannot delete addon: It's currently in use by other records.");
            } else {
                notyf()->error("Database error while deleting addon.");
            }
        } catch (\Exception $e) {
            notyf()->error("Failed to delete addon: " . $e->getMessage());
        }
    }

    public function render()
    {
        $addons = AddOn::query()
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%')
                    ->orWhere('addon_code', 'like', '%' . $this->search . '%')
                    ->orWhere('description', 'like', '%' . $this->search . '%');
            })
            ->when($this->statusFilter !== 'all', function ($query) {
                $query->where('is_active', $this->statusFilter === 'active');
            })
            ->when($this->resourceTypeFilter !== 'all', function ($query) {
                $query->where('resource_type', $this->resourceTypeFilter);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $resourceTypes = AddOn::select('resource_type')->distinct()->pluck('resource_type');

        return view('livewire.admin.plans.addons-list', [
            'addons' => $addons,
            'resourceTypes' => $resourceTypes
        ]);
    }
}
