<?php

namespace App\Livewire\Admin\Plans;

use App\Models\AddOn;
use Livewire\Component;

class AddonForm extends Component
{
    public $addon;
    public $addon_code = '';
    public $name = '';
    public $resource_type = 'storage';
    public $unit = '';
    public $price_per_unit = '';
    public $billing_cycle = 'one-time';
    public $description = '';
    public $is_active = true;
    public $unit_counts = 0;

    public $resourceTypes = [
        'storage' => 'Storage',
        'doctor' => 'Doctor',
        'staff' => 'Staff',
        'api_calls' => 'API Calls',
        'sms' => 'SMS',
        'talecommunication' => 'TaleCommunication',
        'whatsapp' => 'Whatsapp',
    ];

    public $defaultUnits = [
        'storage' => 'GB',
        'doctor' => 'User',
        'staff' => 'User',
        'api_calls' => 'Call',
        'sms' => 'Message',
        'talecommunication' => 'Call',
        'whatsapp' => 'Message'
    ];

    public $billingCycles = [
        'monthly' => 'Monthly',
        'yearly' => 'Yearly',
        'one-time' => 'One Time'
    ];

    public function updatedResourceType($value)
    {
        $this->unit = $this->defaultUnits[$value] ?? '';
    }

    public function updated($propertyName)
    {
        $this->validateOnly($propertyName);
    }

    protected function rules()
    {
        $rules = [
            'addon_code' => 'required|string|max:25|unique:add_ons,addon_code',
            'name' => 'required|string|max:100',
            'resource_type' => 'required|in:storage,doctor,staff,api_calls,sms,talecommunication,whatsapp',
            'unit' => 'required|string|max:20',
            'price_per_unit' => 'required|numeric|min:0',
            'billing_cycle' => 'required|string|max:20',
            'unit_counts' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ];

        if ($this->addon) {
            $rules['addon_code'] = 'required|string|max:25|unique:add_ons,addon_code,' . $this->addon->id;
        }

        return $rules;
    }

    protected $messages = [
        'addon_code.required' => 'The addon code is required.',
        'addon_code.unique' => 'This addon code already exists.',
        'addon_code.max' => 'The addon code cannot exceed 25 characters.',
        'name.required' => 'The name field is required.',
        'name.max' => 'The name cannot exceed 100 characters.',
        'resource_type.required' => 'Please select a resource type.',
        'resource_type.in' => 'Please select a valid resource type.',
        'unit.required' => 'The unit field is required.',
        'unit.max' => 'The unit cannot exceed 20 characters.',
        'price_per_unit.required' => 'The price per unit is required.',
        'price_per_unit.numeric' => 'The price must be a number.',
        'price_per_unit.min' => 'The price cannot be negative.',
        'billing_cycle.required' => 'Please select a billing cycle.',
        'billing_cycle.max' => 'The billing cycle cannot exceed 20 characters.',
        'unit_counts.required' => 'The unit count is required.',
        'unit_counts.numeric' => 'The unit count must be a number.',
        'unit_counts.min' => 'The unit count cannot be negative.',
    ];

    public function mount($addon = null)
    {
        if ($addon) {
            $this->addon = $addon;
            $this->addon_code = $addon->addon_code;
            $this->name = $addon->name;
            $this->resource_type = $addon->resource_type;
            $this->unit = $addon->unit;
            $this->price_per_unit = $addon->price_per_unit;
            $this->billing_cycle = $addon->billing_cycle;
            $this->description = $addon->description;
            $this->is_active = $addon->is_active;
            $this->unit_counts = $addon->unit_counts ?? 0;
        }
    }

    public function save()
    {
        $validatedData = $this->validate();

        if ($this->addon) {
            $this->addon->update($validatedData);
            notyf()->success('Addon updated successfully.');
        } else {
            $addon = AddOn::create($validatedData);
            notyf()->success('Addon created successfully.');
        }

        return redirect()->route('admin.plans.addons.index');
    }

    public function render()
    {
        return view('livewire.admin.plans.addon-form');
    }
}