<?php

namespace App\Livewire\Admin\Patient;

use App\Models\User;
use App\Models\Patient;
use App\Models\PatientClinic;
use Livewire\Component;
use Illuminate\Support\Facades\Log;

class PatientView extends Component
{
    public $patient;
    public $user;
    public $patientClinics;
    public $patientId;
    public $activeTab = 'appointments';

    public function mount($id)
    {
        try {
            if (!is_numeric($id)) {
                throw new \Exception("Invalid patient ID format: " . $id);
            }

            $this->user = User::with([
                'patient',
                'patientClinics.clinic',
                'patientClinics.doctor.doctor'
            ])->find($id);

            if (!$this->user || !$this->user->hasRole('patient')) {
                throw new \Exception("Patient not found or user is not a patient");
            }

            $this->patient = $this->user->patient;
            $this->patientClinics = $this->user->patientClinics()
                ->with([
                    'clinic',
                    'doctor.doctor'
                ])
                ->get();
            $this->patientId = $id;
        } catch (\Exception $e) {
            Log::error('PatientView mount failed:', ['error' => $e->getMessage(), 'id' => $id]);
            session()->flash('error', 'Failed to load patient data: ' . $e->getMessage());
            return redirect()->route('admin.patient-list');
        }
    }

    public function setActiveTab($tab)
    {
        $this->activeTab = $tab;
    }

    public function getInitials($name)
    {
        if (empty($name)) {
            return 'NA';
        }

        $words = explode(' ', trim($name));

        if (count($words) >= 2) {
            return strtoupper(substr($words[0], 0, 1) . substr($words[count($words) - 1], 0, 1));
        } else {
            return strtoupper(substr($name, 0, 2));
        }
    }

    public function getAvatarColorClass($name)
    {
        $colors = ['bg-primary', 'bg-success', 'bg-info', 'bg-warning', 'bg-danger', 'bg-secondary'];
        $hash = crc32($name);
        return $colors[abs($hash) % count($colors)];
    }

    public function getAge()
    {
        if (!$this->patient || !$this->patient->date_of_birth) {
            return null;
        }
        return $this->patient->date_of_birth->age;
    }

    public function editPatient()
    {
        return redirect()->route('admin.patient-management', ['id' => $this->patientId]);
    }

    public function backToList()
    {
        return redirect()->route('admin.patient-list');
    }

    public function bookAppointment()
    {
        return redirect()->route('appointments.create', ['patient_id' => $this->patientId]);
    }

    public function render()
    {
        return view('livewire.admin.patient.patient-view');
    }
}
