<?php

namespace App\Livewire\Admin\Patient;

use Livewire\Component;
use Livewire\WithFileUploads; // Add this trait
use App\Models\User;
use App\Models\Patient;
use App\Models\PatientClinic;
use App\Models\Clinic;
use App\Mail\Patient\CreatePatientMail;
use App\Mail\Patient\UpdatePatientMail;
use Illuminate\Support\Facades\Mail;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class PatientManagement extends Component
{
    use WithFileUploads; // Add this trait for file uploads

    public $name, $email, $password, $password_confirmation;
    public $address, $phone, $date_of_birth, $gender, $medical_history;
    public $status = 'active';
    public $editMode = false;
    public $userIdBeingEdited = null;

    public $country;
    public $state;
    public $city;

    public $doctors = [];
    public $selectedDoctor;
    public $selectedClinic;
    public $clinics = [];
    public $notes = '';

    public $patientMode = 'create';
    public $existingUsers = [];
    public $selectedExistingUser;
    public $searchTerm = '';
    public $dropdownVisible = false;

    // Add profile image properties
    public $profileImage;
    public $currentProfileImage;
    protected $listeners = ['dateChanged'];

    public function dateChanged($value)
    {
        $this->date_of_birth = $value;
        // Optional: Add validation or logging here
        //logger()->info('Date of birth updated to: ' . $value);
    }

    protected function rules()
    {
        $rules = [
            'status' => 'required|in:active,inactive',
            'selectedDoctor' => 'required|exists:users,id',
            'selectedClinic' => 'nullable|exists:clinics,id',
            'notes' => 'nullable|string',
            'address' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'date_of_birth' => 'required|date_format:d-m-Y',
            'gender' => 'required|in:male,female,other',
            'medical_history' => 'nullable|string',
            'country' => 'nullable|string',
            'state' => 'nullable|string',
            'city' => 'nullable|string',
            'profileImage' => 'nullable|image|max:2048', // 2MB max
        ];

        if ($this->patientMode === 'select') {
            $rules['selectedExistingUser'] = 'required|exists:users,id';
            $rules['name'] = 'nullable|string|max:255';
            $rules['email'] = 'nullable|email|max:255';
            $rules['phone'] = 'nullable|string|max:20';
        } else {
            $rules['name'] = 'required|string|max:255';
            $rules['email'] = 'required|email|max:255|unique:users,email' . ($this->userIdBeingEdited ? ',' . $this->userIdBeingEdited : '');
            if (!$this->editMode) {
                $rules['password'] = 'required|min:6|confirmed';
            } elseif ($this->password) {
                $rules['password'] = 'required|min:6|confirmed';
            }
        }

        return $rules;
    }

    public function mount()
    {
        $this->loadDoctors();
        $this->loadExistingUsers();

        $routeId = request()->route('id');
        $queryEditId = request()->get('edit_id');
        $queryId = request()->get('id');

        $editId = $routeId ?? $queryEditId ?? $queryId;

        if ($editId) {
            $this->edit($editId);
        }
    }

    public function togglePatientMode($mode)
    {
        $this->resetValidation();
        $this->patientMode = $mode;

        if ($mode === 'create') {
            $this->selectedExistingUser = null;
            $this->reset(['name', 'email']);
        }
    }

    public function updatedSearchTerm()
    {
        $this->dropdownVisible = !empty($this->searchTerm);
        $this->loadExistingUsers();
    }

    public function loadExistingUsers()
    {
        $query = User::query()
            ->where('id', '!=', auth()->id())
            ->whereDoesntHave('roles', function ($q) {
                $q->where('name', 'patient');
            })
            ->orderBy('name');

        if (!empty($this->searchTerm)) {
            $query->where(function ($q) {
                $q->where('name', 'like', '%' . $this->searchTerm . '%')
                    ->orWhere('email', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->existingUsers = $query->take(100)->get();
    }

    public function selectUser($userId)
    {
        $this->selectedExistingUser = $userId;
        $this->dropdownVisible = false;
        $this->searchTerm = '';
    }

    public function updatedSelectedExistingUser($value)
    {
        if ($value) {
            $user = User::find($value);
            if ($user) {
                $this->name = $user->name;
                $this->email = $user->email;
                $this->phone = $user->phone_number ?? '';
            }
        }
    }

    protected function loadDoctors()
    {
        $this->doctors = User::whereHas('roles', function ($query) {
            $query->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
        })
            ->with(['clinics' => function ($query) {
                $query->where('clinics.status', 'active')
                    ->where('doctor_clinic.status', 'active');
            }])
            ->where('users.status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get()
            ->map(function ($doctor) {
                return [
                    'id' => $doctor->id,
                    'name' => $doctor->name,
                    'email' => $doctor->email,
                    'clinics_count' => $doctor->clinics->count()
                ];
            })
            ->toArray();
    }

    public function updatedSelectedDoctor($doctorId)
    {
        $this->reset(['clinics']);
        $currentClinic = $this->selectedClinic;

        if ($doctorId) {
            $this->loadClinicsForDoctor($doctorId);

            // Check if the current clinic is still valid for the new doctor
            if ($currentClinic && $this->editMode) {
                $clinicExists = collect($this->clinics)->contains('id', $currentClinic);
                if ($clinicExists) {
                    $this->selectedClinic = $currentClinic;
                } else {
                    $this->selectedClinic = null;
                }
            }
        } else {
            $this->selectedClinic = null;
        }
    }

    // Handle profile image upload
    protected function handleProfileImageUpload()
    {
        if ($this->profileImage) {
            $filename = 'patients/' . uniqid() . '.' . $this->profileImage->getClientOriginalExtension();
            // Store in public disk and return the path relative to storage/app/public
            $path = $this->profileImage->storeAs('patients', uniqid() . '.' . $this->profileImage->getClientOriginalExtension(), 'public');
            return $path;
        }
        return null;
    }

    // Remove old profile image
    protected function removeOldProfileImage($imagePath)
    {
        if ($imagePath && Storage::disk('public')->exists($imagePath)) {
            Storage::disk('public')->delete($imagePath);
        }
    }

    public function createPatient()
    {
        $this->validate();

        try {
            DB::transaction(function () {
                if ($this->patientMode === 'select') {
                    $this->assignExistingUserAsPatient();
                } else {
                    $this->createNewPatient();
                }
            });

            notyf()->success('Patient created successfully.');
            $this->resetForm();
            return redirect()->route('admin.patient-list');
        } catch (\Exception $e) {
            Log::error('Patient creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create patient: ' . $e->getMessage());
        }
    }

    protected function assignExistingUserAsPatient()
    {
        $user = User::findOrFail($this->selectedExistingUser);

        if ($user->hasRole('patient')) {
            $existingAssignment = PatientClinic::where('patient_id', $user->id)
                ->where('clinic_id', $this->selectedClinic)
                ->where('doctor_id', $this->selectedDoctor)
                ->exists();

            if ($existingAssignment) {
                throw new \Exception('This user is already assigned as a patient to this clinic under the selected doctor.');
            }
        } else {
            $patientRole = Role::where('name', 'patient')
                ->where('guard_name', 'web')
                ->firstOrFail();
            $user->assignRole($patientRole);
        }

        $profileImagePath = $this->handleProfileImageUpload();

        $patient = Patient::firstOrCreate(
            ['user_id' => $user->id],
            [
                'address' => $this->address,
                'phone' => $this->phone,
                'date_of_birth' => $this->date_of_birth,
                'gender' => $this->gender,
                'medical_history' => $this->medical_history,
                'profile_image' => $profileImagePath,
            ]
        );

        if (!$patient->wasRecentlyCreated) {
            $updateData = [
                'address' => $this->address ?: $patient->address,
                'phone' => $this->phone ?: $patient->phone,
                'date_of_birth' => $this->date_of_birth ?: $patient->date_of_birth,
                'gender' => $this->gender ?: $patient->gender,
                'medical_history' => $this->medical_history ?: $patient->medical_history,
            ];

            if ($profileImagePath) {
                $this->removeOldProfileImage($patient->profile_image);
                $updateData['profile_image'] = $profileImagePath;
            }

            $patient->update($updateData);
        }

        if ($this->selectedDoctor && $this->selectedClinic) {
            $this->createPatientClinicAssociation($patient);
        }
    }

    public function resetForm()
    {
        $this->reset([
            'name',
            'email',
            'password',
            'password_confirmation',
            'address',
            'phone',
            'date_of_birth',
            'gender',
            'medical_history',
            'status',
            'userIdBeingEdited',
            'editMode',
            'selectedDoctor',
            'selectedClinic',
            'clinics',
            'notes',
            'patientMode',
            'existingUsers',
            'selectedExistingUser',
            'searchTerm',
            'dropdownVisible',
            'profileImage',
            'currentProfileImage'
        ]);
        $this->resetValidation();
    }

    protected function createNewPatient()
    {
        if (User::where('email', $this->email)->exists()) {
            throw new \Exception('Email already exists. Please use a different email.');
        }
        try {
            $user = User::create([
                'name' => $this->name,
                'email' => $this->email,
                'password' => Hash::make($this->password),
                'status' => $this->status,
            ]);

            $patientRole = Role::where('name', 'patient')
                ->where('guard_name', 'web')
                ->firstOrFail();

            $user->assignRole($patientRole);

            $profileImagePath = $this->handleProfileImageUpload();

            $patient = Patient::create([
                'user_id' => $user->id,
                'address' => $this->address,
                'phone' => $this->phone,
                'date_of_birth' => $this->date_of_birth,
                'gender' => $this->gender,
                'medical_history' => $this->medical_history,
                'profile_image' => $profileImagePath,
            ]);

            if ($this->selectedDoctor && $this->selectedClinic) {
                $this->createPatientClinicAssociation($patient);
            }

            #Mail::to($user->email)->send(new CreatePatientMail());
            #Log::info("✅ CreatePatientMail sent to {$user->email}");

            $this->resetForm();
            return redirect()->route('admin.patient-list');
        } catch (\Exception $e) {
            Log::error('Patient creation failed: ' . $e->getMessage());
            notyf()->error('Failed to create patient: ' . $e->getMessage());
        }
    }

    public function updatePatient()
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $this->userIdBeingEdited,
            'status' => 'required|in:active,inactive',
            'address' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'medical_history' => 'nullable|string',
            'selectedDoctor' => 'nullable|exists:users,id',
            'selectedClinic' => 'nullable|exists:clinics,id',
            'notes' => 'nullable|string',
            'profileImage' => 'nullable|image|max:2048',
        ];

        if ($this->password) {
            $rules['password'] = 'required|min:6|confirmed';
        }

        $this->validate($rules);

        try {
            $user = User::findOrFail($this->userIdBeingEdited);
            $patient = $user->patient;

            $updateData = [
                'name' => $this->name,
                'email' => $this->email,
                'status' => $this->status,
            ];

            if ($this->password) {
                $updateData['password'] = Hash::make($this->password);
            }

            $user->update($updateData);

            $patientUpdateData = [
                'address' => $this->address,
                'phone' => $this->phone,
                'date_of_birth' => $this->date_of_birth,
                'gender' => $this->gender,
                'medical_history' => $this->medical_history,
            ];

            // Handle profile image update
            if ($this->profileImage) {
                $this->removeOldProfileImage($patient->profile_image);
                $patientUpdateData['profile_image'] = $this->handleProfileImageUpload();
            }

            $patient->update($patientUpdateData);

            if ($this->selectedDoctor && $this->selectedClinic) {
                $this->updatePatientClinicAssociation($patient);
            }

            #Mail::to($user->email)->send(new UpdatePatientMail());
            #Log::info("✅ UpdatePatientMail sent to {$user->email}");

            notyf()->success('Patient updated successfully.');
            $this->resetForm();
        } catch (\Exception $e) {
            Log::error('Patient update failed: ' . $e->getMessage());
            notyf()->error('Failed to update patient: ' . $e->getMessage());
        }
    }

    protected function createPatientClinicAssociation(Patient $patient)
    {
        PatientClinic::create([
            'patient_id' => $patient->user_id,
            'clinic_id' => $this->selectedClinic,
            'doctor_id' => $this->selectedDoctor,
            'added_by' => Auth::id(),
            'status' => 'active',
            'approval_status' => 'approved',
            'approved_at' => now(),
            'notes' => $this->notes ? [$this->notes] : null
        ]);
    }

    protected function updatePatientClinicAssociation(Patient $patient)
    {
        // First deactivate any existing active associations
        PatientClinic::where('patient_id', $patient->user_id)
            ->where('status', 'active')
            ->update(['status' => 'inactive']);

        // Create or update the new association
        PatientClinic::updateOrCreate(
            [
                'patient_id' => $patient->user_id,
                'clinic_id' => $this->selectedClinic,
                'doctor_id' => $this->selectedDoctor
            ],
            [
                'added_by' => Auth::id(),
                'status' => 'active',
                'approval_status' => 'approved',
                'approved_at' => now(),
                'notes' => $this->notes ? [$this->notes] : null
            ]
        );
    }

    public function edit($id)
    {
        try {
            if (!is_numeric($id)) {
                throw new \Exception("Invalid ID format: " . $id);
            }

            $user = User::with(['patient.patientClinics'])->find($id);

            if (!$user || !$user->hasRole('patient')) {
                throw new \Exception("Patient not found");
            }

            $this->editMode = true;
            $this->userIdBeingEdited = $id;
            $this->name = $user->name;
            $this->email = $user->email;
            $this->status = $user->status;
            $this->password = '';
            $this->password_confirmation = '';

            if ($user->patient) {
                $this->address = $user->patient->address;
                $this->phone = $user->patient->phone;
                $this->date_of_birth = $user->patient->date_of_birth ?
                    $user->patient->date_of_birth->format('d-m-Y') : '';
                $this->gender = $user->patient->gender;
                $this->medical_history = $user->patient->medical_history;
                $this->currentProfileImage = $user->patient->profile_image;

                $patientClinic = PatientClinic::where('patient_id', $user->id)
                    ->where('status', 'active')
                    ->latest()
                    ->first();

                if ($patientClinic) {
                    $this->selectedDoctor = $patientClinic->doctor_id;
                    $this->selectedClinic = $patientClinic->clinic_id;
                    $this->notes = is_array($patientClinic->notes) ? implode(', ', $patientClinic->notes) : ($patientClinic->notes ?? '');

                    $this->loadClinicsForDoctor($patientClinic->doctor_id);
                }
            }

            $this->dispatch('scroll-to-form');
        } catch (\Exception $e) {
            Log::error('Edit failed:', ['error' => $e->getMessage(), 'id' => $id]);
            notyf()->error('Failed to load patient data: ' . $e->getMessage());
        }
    }

    protected function loadClinicsForDoctor($doctorId)
    {
        $this->clinics = [];

        if ($doctorId) {
            $doctor = User::with(['clinics' => function ($query) {
                $query->where('clinics.status', 'active')
                    ->where('doctor_clinic.status', 'active');
            }])->find($doctorId);

            if ($doctor && $doctor->clinics->count() > 0) {
                $this->clinics = $doctor->clinics->map(function ($clinic) {
                    return [
                        'id' => $clinic->id,
                        'name' => $clinic->name,
                        'address' => $clinic->address ?? '',
                        'is_primary' => $clinic->pivot->is_primary ?? false
                    ];
                })->toArray();
            }
        }
    }

    public function save()
    {
        if ($this->editMode) {
            $this->updatePatient();
        } else {
            $this->createPatient();
        }
    }

    public function render()
    {
        return view('livewire.admin.patient.patient-management');
    }
}
