<?php

namespace App\Livewire\Admin\Patient;

use App\Models\User;
use Illuminate\Support\Facades\Gate;
use Livewire\Component;
use Livewire\WithPagination;

class PatientList extends Component
{
    use WithPagination;

    public $searchTerm = '';
    public $statusFilter = 'all';
    public $genderFilter = 'all';
    public $doctorFilter = 'all';
    public $perPage = 10;
    public $sortBy = 'recent';

    protected $queryString = [
        'searchTerm' => ['except' => ''],
        'statusFilter' => ['except' => 'all'],
        'genderFilter' => ['except' => 'all'],
        'doctorFilter' => ['except' => 'all'],
        'perPage' => ['except' => 10],
        'sortBy' => ['except' => 'recent'],
    ];

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $patients = User::whereHas('roles', function ($q) {
            $q->where('name', 'patient');
        })
            ->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', $searchTerm)
                    ->orWhere('email', 'like', $searchTerm);
            })
            ->with(['patient', 'patientClinics.doctor.doctor'])
            ->when($this->statusFilter !== 'all', function ($q) {
                $q->where('status', $this->statusFilter);
            })
            ->when($this->genderFilter !== 'all', function ($q) {
                $q->whereHas('patient', function ($q) {
                    $q->where('gender', $this->genderFilter);
                });
            })
            ->when($this->doctorFilter !== 'all', function ($q) {
                $q->whereHas('patientClinics', function ($q) {
                    $q->where('doctor_id', $this->doctorFilter)
                        ->where('status', 'active');
                });
            })
            ->when($this->sortBy === 'recent', function ($q) {
                $q->orderBy('created_at', 'desc');
            })
            ->when($this->sortBy === 'oldest', function ($q) {
                $q->orderBy('created_at', 'asc');
            })
            ->paginate($this->perPage);

        $doctors = User::whereHas('roles', function ($q) {
            $q->where('group', 'doctor_group')
                ->where('name', '!=', 'doctorstaff');
        })
            ->with('doctor')
            ->get();

        return view('livewire.admin.patient.patient-list', [
            'patients' => $patients,
            'totalPatients' => User::whereHas('roles', function ($q) {
                $q->where('name', 'patient');
            })->count(),
            'doctors' => $doctors
        ]);
    }

    public function resetFilters()
    {
        $this->reset([
            'statusFilter',
            'genderFilter',
            'doctorFilter'
        ]);
    }

    public function toggleStatus($id)
    {
        Gate::authorize('edit patients');
        $user = User::findOrFail($id);
        $patient = $user->patient;
        $patient->status = $patient->status === 'active' ? 'inactive' : 'active';
        $patient->save();

        notyf()->success('Patient status updated successfully.');
    }

    public function deletePatient($id)
    {
        if (auth()->id() == $id) {
            notyf()->error('You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        if ($user->patient) {
            $user->patient->delete();
        }

        $user->delete();

        notyf()->success('Patient deleted successfully.');
    }

    public function sortBy($type)
    {
        $this->sortBy = $type;
        $this->resetPage();
    }
}
