<?php

namespace App\Livewire\Admin\Clinics;

use App\Models\Clinic;
use App\Models\User;
use Livewire\Component;
use Illuminate\Support\Facades\Gate;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Log;

class DoctorsTable extends Component
{
    use WithPagination;
    public Clinic $clinic;
    public $sortField = 'name';
    public $sortDirection = 'asc';
    public $isLoaded = false; // Track if data is loaded
    public $shouldLoad = false; // Control when to load data
    // New properties you need to add
    public $searchTerm = '';
    public $statusFilter = 'all';
    public $approvalStatusFilter = 'all';
    public $perPage = 10;
    public $deleteId = null;
    public $showDeleteModal = false;
    public $doctorIdToDelete;
    public $clinicIdToDelete;
    public $deleteConfirmation = '';
    protected $listeners = [
        'confirm-delete-doctor' => 'confirmDeleteDoctor',
    ];

    public function openDeleteModal($doctorId, $clinicId)
    {
        $this->dispatch(
            'show-delete-modal',
            doctorId: $doctorId,
            clinicId: $clinicId
        );
    }
    public function confirmDeleteDoctor($doctorId, $clinicId)
    {
        if (!$doctorId || !$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'Invalid request.');
            return;
        }

        $this->deleteDoctor($doctorId, $clinicId);
        $this->dispatch('hide-delete-modal');
    }

    public function updatedSearchTerm()
    {
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function applyFilters()
    {
        $this->resetPage();
    }

    public function clearAllFilters()
    {
        $this->statusFilter = 'all';
        $this->approvalStatusFilter = 'all';
        $this->searchTerm = '';
        $this->resetPage();
    }

    public function resetStatusFilter()
    {
        $this->statusFilter = 'all';
        $this->resetPage();
    }

    public function resetAprovelStatusFilter()
    {
        $this->approvalStatusFilter = 'all';
        $this->resetPage();
    }

    public function mount(Clinic $clinic, $loadImmediately = false)
    {
        $this->clinic = $clinic;
        $this->shouldLoad = $loadImmediately;

        if ($this->shouldLoad) {
            $this->loadData();
        }
    }

    #[On('tabSwitched')]
    public function handleTabSwitch($tab)
    {
        if ($tab === 'doctors' && !$this->isLoaded) {
            $this->loadData();
        }
    }

    public function loadData()
    {
        if (!$this->isLoaded) {
            $this->resetPage();
            $this->isLoaded = true;
        }
    }

    public function sortBy($field)
    {
        // Load data if not already loaded
        if (!$this->isLoaded) {
            $this->loadData();
        }

        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
        $this->resetPage();
    }

    public function toggleStatus($id, $clinicId)
    {
        $user = User::findOrFail($id);
        $clinic = $user->clinicAssociations()->where('clinic_id', $clinicId)->firstOrFail();
        $clinic->status = $clinic->status === 'active' ? 'inactive' : 'active';
        $clinic->save();

        $this->resetPage();
        notyf()->success('Doctor Clinic status updated successfully.');
    }

    public function toggleApprovalStatus($id, $clinicId, $status)
    {
        $user = User::findOrFail($id);
        $clinic = $user->clinicAssociations()->where('clinic_id', $clinicId)->firstOrFail();
        $clinic->approval_status = $status;

        if ($clinic->approval_status === 'rejected' || $clinic->approval_status === 'pending') {
            $clinic->status = 'inactive';
        } elseif ($clinic->approval_status === 'approved') {
            $clinic->status = 'active';
        }

        $clinic->save();
        $this->resetPage();
        notyf()->success('Doctor Clinic Approval Status updated successfully.');
    }

    public function deleteDoctor($id, $clinicId)
    {
        Gate::authorize('delete doctors');
        Log::info($id);
        Log::info($clinicId);


        if (auth()->id() == $id) {
            $this->dispatch('alert', type: 'error', message: 'You cannot delete your own account.');
            return;
        }

        $user = User::findOrFail($id);

        if (!$user->isDoctor()) {
            $this->dispatch('alert', type: 'error', message: 'The selected user is not a doctor.');
            return;
        }

        if (!$clinicId) {
            $this->dispatch('alert', type: 'error', message: 'No clinic selected.');
            return;
        }

        $user->clinics()->detach($clinicId);
        $user->clinicAssociations()->where('clinic_id', $clinicId)->delete();
        $this->deleteId = null;

        $this->resetPage();
        notyf()->success('Doctor removed from clinic successfully.');
    }

    public function render()
    {
        $searchTerm = '%' . $this->searchTerm . '%';

        $doctors = User::where(function ($q) use ($searchTerm) {
            $q->where('name', 'like', $searchTerm)
                ->orWhere('email', 'like', $searchTerm)
                ->orWhereHas('doctorProfile', function ($subQ) use ($searchTerm) {
                    $subQ->where('license_number', 'like', $searchTerm);
                });
        })
            ->whereHas('roles', function ($q) {
                $q->where('group', 'doctor_group')
                    ->where('name', '!=', 'doctorstaff');
            })
            ->whereHas('clinics', function ($q) {
                $q->where('clinics.id', $this->clinic->id);
            })
            ->when($this->statusFilter !== 'all', function ($q) {
                $q->whereHas('clinics', function ($subQ) {
                    $subQ->where('clinics.id', $this->clinic->id)
                        ->where('doctor_clinic.status', $this->statusFilter);
                });
            })
            ->when($this->approvalStatusFilter !== 'all', function ($q) {
                $q->whereHas('clinics', function ($subQ) {
                    $subQ->where('clinics.id', $this->clinic->id)
                        ->where('doctor_clinic.approval_status', $this->approvalStatusFilter);
                });
            })
            ->with([
                'clinicAssociations' => function ($q) {
                    $q->where('clinic_id', $this->clinic->id);
                },
                'doctorProfile',
                'roles',
                'clinics' => function ($q) {
                    $q->where('clinics.id', $this->clinic->id)
                        ->withPivot(['status', 'approval_status', 'is_primary']);
                }
            ])
            ->when($this->sortField === 'name', function ($q) {
                $q->orderBy('name', $this->sortDirection);
            })
            ->when($this->sortField === 'email', function ($q) {
                $q->orderBy('email', $this->sortDirection);
            })
            ->when($this->sortField === 'license_number', function ($q) {
                $q->join('doctor_profiles', 'users.id', '=', 'doctor_profiles.user_id')
                    ->orderBy('doctor_profiles.license_number', $this->sortDirection);
            })
            ->when($this->sortField === 'status', function ($q) {
                $q->join('doctor_clinic', 'users.id', '=', 'doctor_clinic.doctor_id')
                    ->where('doctor_clinic.clinic_id', $this->clinic->id)
                    ->orderBy('doctor_clinic.status', $this->sortDirection);
            })
            ->paginate($this->perPage);

        return view('livewire.admin.clinics.doctors-table', [
            'doctors' => $doctors,
        ]);
    }
}