<?php

namespace App\Http\Controllers\Patient;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Appointment;
use App\Models\User;
use App\Models\Doctor;
use App\Models\Clinic;
use App\Models\Specialization;
use Carbon\Carbon; // ✅ Add this line

use App\Models\Patient;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;


class AppointmentController extends Controller
{
    /**
     * Show the list of patient appointments.
     */
    public function index()
    {
        // Fetch appointments for the logged-in patient
        // ✅ Get logged-in patient ID
        $loggedInPatientId = Auth::id();

        // ✅ Fetch appointments for this patient
        $appointments = Appointment::with([
            'doctor:id,name,email', // Doctor info
        ])
        ->where('patient_id', $loggedInPatientId)
        ->orderBy('id', 'DESC')
        ->get();

        // ✅ Fetch all doctors linked to the patient’s appointments (for extra details like specialization)
        $doctorIds = $appointments->pluck('doctor_id')->unique();
        $doctors = Doctor::whereIn('id', $doctorIds)->get();

        // ✅ Pass data to view
        return view('patient.appointments.appointment-list', compact('appointments', 'doctors')); // Blade: resources/views/patient/appointments/list.blade.php
    }

    public function destroy($id)
    {
        $appointment = Appointment::find($id);

        if (!$appointment) {
            return redirect()->back()->with('error', 'Appointment not found.');
        }

        $appointment->delete();

        return redirect()->back()->with('success', 'Appointment deleted successfully.');
    }

    public function show($id)
    {
        $appointment = Appointment::with(['doctor', 'patient']) // make sure relationships exist
            ->findOrFail($id);

        return response()->json($appointment);
    }
    /**
     * Show the form to create a new appointment.
     */
    public function create()
    {
        // Fetch required data like doctors list, available slots, etc.
        // Example: $doctors = Doctor::all();

        // ✅ Fetch active specializations
        $specializations = Specialization::where('status', 1)
            ->orderBy('specialization', 'asc')
            ->get();

        // ✅ Logged-in Patient
        $loggedInPatientId = auth()->user()->id;
        $loggedInPatient = User::find($loggedInPatientId);

        return view('patient.appointments.appointment-form', compact('loggedInPatient', 'specializations')); // Blade: resources/views/patient/appointments/create.blade.php
    }

    public function getDoctorsBySpecializationName($specialization)
    {
        $doctors = DB::table('doctors')
            ->join('users', 'doctors.user_id', '=', 'users.id')
            ->where('doctors.specialization', 'LIKE', '%' . $specialization . '%')
            ->whereNull('users.deleted_at')
            ->select('users.id', 'users.name')
            ->get();

        if ($doctors->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No doctors found for this specialization.'
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $doctors
        ]);
    }

    public function getSlots(Request $request, $id)
    {
        $date = $request->query('date');

        if (!$date) {
            return response()->json(['success' => false, 'message' => 'Date is required']);
        }

        $dayOfWeek = Carbon::createFromFormat('Y-m-d', $date)->format('l');

        // Step 1: Get doctor's availability for that day
        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $id)
            ->where('status', 1)
            ->get(['time_slot', 'day']);

        $availableSlots = [];

        foreach ($availability as $item) {
            $days = json_decode($item->day, true);

            if (is_array($days) && in_array($dayOfWeek, $days)) {
                $timeSlots = json_decode($item->time_slot, true);

                if (is_array($timeSlots)) {
                    $availableSlots = array_merge($availableSlots, $timeSlots);
                }
            }
        }

        $bookedSlotsRaw = Appointment::where('doctor_id', $id)
        ->whereDate('appointment_date', $date)
        ->pluck('appointment_slot_time')
        ->toArray();

        // Decode all JSON strings inside the array and flatten
        $bookedSlots = [];

        foreach ($bookedSlotsRaw as $jsonString) {
            $slots = json_decode($jsonString, true);
            if (is_array($slots)) {
                $bookedSlots = array_merge($bookedSlots, $slots);
            }
        }

        // Step 3: Match available and booked slots
        $finalSlots = [];

        foreach ($availableSlots as $slot) {
            $slotFormatted = trim($slot);
            $isBooked = in_array($slotFormatted, $bookedSlots);
            $finalSlots[] = [
                'time' => $slotFormatted,
                'status' => $isBooked ? 'booked' : 'available'
            ];
        }

        return response()->json(['success' => true, 'slots' => $finalSlots]);
    }

    public function getAvailability($id)
    {
        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $id)
            ->where('status', 1)
            ->get();

        if ($availability->isEmpty()) {
            return response()->json(['success' => false, 'message' => 'No availability found']);
        }

        $slots = [];
        foreach ($availability as $item) {
            $times = json_decode($item->time_slot, true); // JSON column
            if (is_array($times)) {
                foreach ($times as $slot) {
                    $slots[] = $slot;
                }
            }
        }

        return response()->json(['success' => true, 'slots' => $slots]);
    }

    public function getFee($id)
    {
        $availability = DB::table('doctor_availability')
        ->where('doctor_id', $id)
        ->first();

        if ($availability) {
            $timeSlots = json_decode($availability->time_slot, true) ?? [];

            return response()->json([
                'success' => true,
                'slots' => $timeSlots,
                'in_person_fee' => $availability->in_person_fee,
                'video_fee' => $availability->video_fee,
            ]);
        }

        return response()->json(['success' => false]);
    }
}
