<?php

namespace App\Http\Controllers\Patient\Api\V1;
use Laravel\Sanctum\PersonalAccessToken;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;  
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

use App\Models\ActiveDevice;
use App\Models\Clinic;
use App\Models\Patient;
use App\Models\PatientClinic;
use App\Models\User; // Assuming Patient is stored in users table with 'patient' role

class PatientApiAuthController extends Controller
{
    // Send OTP API
    public function sendOtp(Request $request)
    {
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
        ]);

        if (!$request->email && !$request->phone_number) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or phone number is required.'
            ], 422);
        }

        // ✅ Check if user exists based on email or phone
        if ($request->email) {
            $user = User::where('email', $request->email)->first();
            if ($user) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'This email is already registered.'
                ], 400);
            }
        }

        if ($request->phone_number) {
            $user = User::where('phone_number', $request->phone_number)->first();
            if ($user) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'This phone number is already registered.'
                ], 400);
            }
        }

        // ✅ Static OTP for now
        $otp = '1234';

        // Normally: Save OTP in DB or cache for verification

        return response()->json([
            'status'  => 'success',
            'message' => 'OTP sent successfully for registration.',
            'otp'     => $otp // ✅ For testing only
        ]);
    }

    // Verify OTP API
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'phone_number' => 'required|string|min:10|max:15',
            'otp'          => 'required|string'
        ]);

        // ✅ Static OTP Check
        if ($request->otp !== '1234') {
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid OTP.'
            ], 400);
        }

        return response()->json([
            'status'  => 'success',
            'message' => 'OTP verified successfully.'
        ]);
    }

    /**
     * ✅ Patient Signup
     */
    public function signup(Request $request)
    {
        try {
            // ✅ Custom Validation
            $validator = \Validator::make($request->all(), [
                'name'        => 'required|string|max:255',
                'email'       => 'required|email|unique:users,email',
                'contact'     => 'required|string|max:20|unique:users,phone_number',
                'city'        => 'required|string|max:100',
                'device_name' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Validation failed.',
                    'errors'  => $validator->errors()
                ], 422);
            }

            DB::beginTransaction();

            // ✅ Create User
            $user = User::create([
                'name'         => $request->name,
                'email'        => $request->email,
                'phone_number' => $request->contact
            ]);

            // ✅ Assign Role
            $user->assignRole('patient');

            // ✅ Create Patient Profile
            $patient = Patient::create([
                'user_id' => $user->id,
                'address' => $request->city,
                'phone'   => $request->contact
            ]);

            // ✅ Generate Sanctum Token
            $token = $user->createToken($request->device_name)->plainTextToken;

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Patient registered and logged in successfully.',
                'patient_id' => $user->id,
                'access_token' => $token,
                'token_type'   => 'Bearer',
                // 'data'    => [
                //     'user' => [
                //         
                //         'name'    => $user->name,
                //         'email'   => $user->email,
                //         'contact' => $user->phone_number,
                //         'city'    => $request->city
                //     ],
                //     'access_token' => $token,
                //     'token_type'   => 'Bearer'
                // ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Patient Registration Error: ' . $e->getMessage());
            return response()->json([
                'status'  => 'error',
                'message' => 'Failed to register patient.',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * ✅ Send OTP for Login
     */
    public function loginOtp(Request $request)
    {
        $request->validate([
            'email'         => 'nullable|email',
            'phone_number'  => 'nullable|string|min:10|max:15',
        ]);

        if (!$request->email && !$request->phone_number) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or phone number is required.'
            ], 422);
        }

        // ✅ Find user based on which field is provided
        if ($request->email) {
            $user = User::where('email', $request->email)->first();
        } else {
            $user = User::where('phone_number', $request->phone_number)->first();
        }

        if (!$user) {
            return response()->json([
                'status'  => 'error',
                'message' => 'User not registered. Please signup first.'
            ], 404);
        }

        // ✅ Static OTP (for now)
        $otp = '1234';

        return response()->json([
            'status'  => 'success',
            'message' => 'OTP sent successfully for login.',
            'otp'     => $otp // ✅ Only for testing
        ], 200);
    }

    /**
     * ✅ Patient Login
     */
    public function login(Request $request)
    {
        // ✅ Validate केवल email या phone के लिए
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
            'otp'          => 'required|string|min:4|max:6',
        ]);

        // ✅ OTP Check
        if ($request->otp !== '1234') {
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid OTP.'
            ], 400);
        }

        // ✅ Check email या phone से user find करो
        if (!empty($request->email)) {
            $user = User::where('email', $request->email)->first();
        } elseif (!empty($request->phone_number)) {
            $user = User::where('phone_number', $request->phone_number)->first();
        } else {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or Phone number is required.'
            ], 422);
        }

        if (!$user) {
            return response()->json([
                'status'  => 'error',
                'message' => 'User not found. Please register first.'
            ], 404);
        }

        // ✅ Check role (patient fix किया)
        if (!$user->hasRole('patient')) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Access denied.'
            ], 403);
        }

        // ✅ Static device details
        $deviceName = 'MobileApp';
        $deviceId   = '12345';

        // ✅ Create token
        $token = $user->createToken($deviceName)->plainTextToken;

        // ✅ Track Device (optional)
        //$this->trackDevice($user, $request, $token, $deviceId);

        return response()->json([
            'status'  => 'success',
            'message' => 'Logged in successfully',
            'patient_id'      => $user->id, // 👈 yahan se user ki id bhi jayegi
            'access_token' => $token,
            'token_type'   => 'Bearer',
            // 'data'    => [
            //     'user'         => $this->getUserResponse($user),
                
            // ]
        ]);
    }

    // Logout API
    public function logout(Request $request)
    {
        try {
            $request->validate([
                'patient_id' => 'nullable|integer|exists:users,id',
            ]);

            // ✅ CASE 1: Agar patient_id diya ho (force logout all sessions)
            if ($request->filled('patient_id')) {
                $user = User::find($request->patient_id);

                if (!$user) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Patient not found',
                    ], 404);
                }

                // ✅ Saare tokens delete kardo
                PersonalAccessToken::where('tokenable_id', $user->id)
                    ->where('tokenable_type', User::class)
                    ->delete();

                // ✅ ActiveDevice table update (agar use karte ho)
                // ActiveDevice::where('user_id', $user->id)->update([
                //     'is_active' => false,
                //     'logout_at' => now(),
                //     'fcm_token' => null,
                // ]);

                return response()->json([
                    'status' => 'success',
                    'message' => 'Patient logged out from all devices',
                ]);
            }

            // ✅ CASE 2: Agar patient_id nahi diya ho → normal logout (current token)
            $user = $request->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized',
                ], 401);
            }

            $token = $user->currentAccessToken();
            if (!$token) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No active session found',
                ], 400);
            }

            $token->delete();

            // ✅ ActiveDevice table update (agar use karte ho)
            // ActiveDevice::where('user_id', $user->id)
            //     ->where('token_id', $token->id)
            //     ->update([
            //         'is_active' => false,
            //         'logout_at' => now(),
            //         'fcm_token' => null,
            //     ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Successfully logged out',
            ]);

        } catch (\Exception $e) {
            Log::error('Patient Logout error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to logout',
            ], 500);
        }
    }


    // Fetch Profile API
    public function getProfile(Request $request)
    {
        $request->validate([
            'patient_id' => 'required|exists:users,id'
        ]);

        // User + Patient details ek sath fetch
        $user = User::with('patient')->find($request->patient_id);

        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'Patient not found'
            ], 404);
        }

        return response()->json([
            'status'  => 'success',
            'message' => 'Profile details fetched successfully',
            'data'    => [
                'patient_id'    => $user->id,
                'name'          => $user->name,
                'email'         => $user->email,
                'phone_number'  => $user->phone_number,
                'status'        => $user->status,
                'last_login_at' => $user->last_login_at,
                'address'        => $user->patient->address,
                'street_address' => $user->patient->street_address,
                'city'           => $user->patient->city,
                'state'          => $user->patient->state,
                'postal_code'    => $user->patient->postal_code,
                'country'        => $user->patient->country,
                'phone'          => $user->patient->phone,
                'date_of_birth'  => $user->patient->date_of_birth,
                'gender'         => $user->patient->gender,
                'profile_image'  => $user->patient->profile_image 
                                        ? asset('storage/' . $user->patient->profile_image)
                                        : null,
                'medical_history'=> $user->patient->medical_history,

                // 'user' => [
                //     'patient_id'    => $user->id,
                //     'name'          => $user->name,
                //     'email'         => $user->email,
                //     'phone_number'  => $user->phone_number,
                //     'status'        => $user->status,
                //     'last_login_at' => $user->last_login_at,
                // ],
                // 'patient' => $user->patient ? [
                //     'id'             => $user->patient->id,
                //     'status'         => $user->patient->status,
                //     'address'        => $user->patient->address,
                //     'street_address' => $user->patient->street_address,
                //     'city'           => $user->patient->city,
                //     'state'          => $user->patient->state,
                //     'postal_code'    => $user->patient->postal_code,
                //     'country'        => $user->patient->country,
                //     'phone'          => $user->patient->phone,
                //     'date_of_birth'  => $user->patient->date_of_birth,
                //     'gender'         => $user->patient->gender,
                //     'profile_image'  => $user->patient->profile_image 
                //                             ? asset('storage/patients/' . $user->patient->profile_image)
                //                             : null,
                //     'medical_history'=> $user->patient->medical_history,
                //     'verified_at'    => $user->patient->verified_at,
                //     'verified_by'    => $user->patient->verified_by,
                // ] : null
            ]
        ]);
    }

    // Update Profile
    public function updateProfile(Request $request)
    {
        // ✅ Validation
        $request->validate([
            'patient_id'   => 'required|exists:users,id',
            'name'         => 'sometimes|string|max:255',
            'email'        => 'sometimes|email|unique:users,email,' . $request->patient_id,
            'contact'      => 'sometimes|string|max:20|unique:users,phone_number,' . $request->patient_id,
            'city'         => 'sometimes|string|max:100',
            'address'      => 'nullable|string|max:255',
            'dob'          => 'nullable|date',
            'gender'       => 'nullable|string|in:Male,Female,Other',
            'blood_group'  => 'nullable|string',
            'profile_image'=> 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        // ✅ User fetch करो (patient_id से)
        $user = User::find($request->patient_id);
        if (!$user) {
            return response()->json(['status' => 'error', 'message' => 'User not found'], 404);
        }

        // ✅ User update
        $user->update([
            'name'         => $request->name ?? $user->name,
            'email'        => $request->email ?? $user->email,
            'phone_number' => $request->contact ?? $user->phone_number,
        ]);

        // ✅ Patient relation fetch करो
        $patient = $user->patient; // make sure User model में hasOne(Patient::class) है

        // if ($patient) {
        //     // Profile image handle
        //     $profileImagePath = $patient->profile_image; // पुरानी image
        //     if ($request->hasFile('profile_image')) {
        //         // नई image upload करो
        //         $profileImagePath = $request->file('profile_image')->store('patients', 'public');
        //     }

        //     // Patient table update
        //     $patient->update([
        //         'city'         => $request->city ?? $patient->city,
        //         'address'      => $request->address ?? $patient->address,
        //         'dob'          => $request->dob ?? $patient->dob,
        //         'gender'       => $request->gender ?? $patient->gender,
        //         'blood_group'  => $request->blood_group ?? $patient->blood_group,
        //         'profile_image'=> $profileImagePath,
        //     ]);
        // }

        if (!$patient) {
            return response()->json(['success' => false, 'message' => 'Patient not found'], 404);
        }

        return response()->json([
            'success' => true,
            'patient' => [
                'id'            => $patient->id,
                'name'          => $patient->name,
                'city'          => $patient->city,
                'address'       => $patient->address,
                'dob'           => $patient->dob,
                'gender'        => $patient->gender,
                'blood_group'   => $patient->blood_group,
                'profile_image' => $patient->profile_image 
                    ? asset('storage/' . $patient->profile_image) // ✅ full URL generate
                    : null
            ]
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Profile updated successfully',
            // 'data'    => [
            //     'user'    => $user->fresh(),
            //     'patient' => $patient ? $patient->fresh() : null,
            //     'profile_image_url' => $patient && $patient->profile_image ? asset('storage/' . $patient->profile_image) : null,
            // ]
        ]);
    }
}
