<?php

namespace App\Http\Controllers\Doctor;
       
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\Appointment;
use App\Models\User;
use App\Models\Doctor;
use App\Models\Clinic;
use App\Models\Specialization;
use Carbon\Carbon; // ✅ Add this line

use App\Models\Patient;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class AppointmentController extends Controller
{
    public function index()
    {
        //$appointments = Appointment::where('doctor_id', auth()->id())->latest()->get();
    
        // // Fetch all appointments with patient & doctor info
        // $appointments = Appointment::with([
        //     'patient:id,name,email',
        //     'doctor:id,name,email'
        // ])->orderBy('id', 'DESC')->get();

        // return view('doctor.appointments.appointment-list', compact('appointments'));

        $loggedInDoctorId = Auth::id();

        // Step 1: Get the clinic ID for the logged-in doctor
        $clinicRecord = DB::table('doctor_clinic')
            ->where('doctor_id', $loggedInDoctorId)
            ->first();

        $clinicId = $clinicRecord->clinic_id ?? null;
        $doctorIds = [$loggedInDoctorId]; // start with self

        // Step 2: Get other doctor IDs in same clinic (excluding self)
        if ($clinicId) {
            $otherDoctorIds = DB::table('doctor_clinic')
                ->where('clinic_id', $clinicId)
                ->where('doctor_id', '!=', $loggedInDoctorId)
                ->pluck('doctor_id')
                ->toArray();

            $doctorIds = array_merge($doctorIds, $otherDoctorIds);
        }

        // Step 3: Fetch appointments for doctors in this clinic
        $appointments = Appointment::with([
            'patient:id,name,email',
            'doctor:id,name,email',
        ])
        ->whereIn('doctor_id', $doctorIds)
        ->orderBy('id', 'DESC')
        ->get();

        // Step 4: Fetch other doctors in clinic
        $otherDoctors = User::whereIn('id', $doctorIds)
            ->where('id', '!=', $loggedInDoctorId)
            ->whereHas('roles', function ($q) {
                $q->where('name', 'doctor');
            })
            ->get();

        // Step 5: Fetch all patients
        $patients = User::whereHas('roles', function ($q) {
            $q->where('name', 'patient');
        })->get();

        return view('doctor.appointments.appointment-list', compact('appointments', 'otherDoctors', 'patients'));

    }

    // public function create()
    // {
    //     // Fetch active specializations
    //     $specializations = Specialization::where('status', 1)
    //         ->orderBy('specialization', 'asc')
    //         ->get();

    //     $loggedInDoctorId = auth()->user()->id;

    //     // Step 1: Get clinic_id of logged-in doctor
    //     $clinicRecord = DB::table('doctor_clinic')
    //         ->where('doctor_id', $loggedInDoctorId)
    //         ->first();

    //     $loggedInDoctor = User::find($loggedInDoctorId); // <- This was missing

    //     $otherDoctors = collect(); // default empty collection

    //     if ($clinicRecord) {
    //         $clinicId = $clinicRecord->clinic_id;

    //         // Step 2: Get all doctor IDs linked to that clinic (excluding logged-in doctor)
    //         $doctorClinicRecords = DB::table('doctor_clinic')
    //             ->where('clinic_id', $clinicId)
    //             ->where('doctor_id', '!=', $loggedInDoctorId) // exclude self
    //             ->get();

    //         $doctorIds = [];
    //         foreach ($doctorClinicRecords as $record) {
    //             $doctorIds[] = $record->doctor_id;
    //         }

    //         // Step 3: Fetch those doctors
    //         $otherDoctors = User::whereIn('id', $doctorIds)
    //             ->whereHas('roles', function ($query) {
    //                 $query->where('name', 'doctor');
    //             })
    //             ->get();
    //         }

    //         // Get all users with 'patient' role
    //         $patients = User::whereHas('roles', function ($query) {
    //             $query->where('name', 'patient');
    //         })->get();

    //     // $patients = User::where('role', 'patient')->get();
    //     // return view('doctor.appointments.appointment-form', [
    //     //     'patients' => $patients,
    //     //     'appointment' => null
    //     // ]);

    //     return view('doctor.appointments.appointment-form', compact('loggedInDoctor', 'otherDoctors', 'patients', 'specializations', 'clinicRecord'));
    // }


    public function create()
    {
        $loggedInDoctorId = auth()->user()->id;

        // Step 1: Get clinic_id(s) of logged-in doctor
        $clinicRecord = DB::table('doctor_clinic')
            ->where('doctor_id', $loggedInDoctorId)
            ->first();

        $loggedInDoctor = User::find($loggedInDoctorId); // logged-in doctor

        $otherDoctors = collect(); // default empty collection
        $specializations = collect(); // default empty collection

        if ($clinicRecord) {
            $clinicId = $clinicRecord->clinic_id;

            // Step 2: Get all doctor IDs linked to that clinic (excluding logged-in doctor)
            $doctorClinicRecords = DB::table('doctor_clinic')
                ->where('clinic_id', $clinicId)
                ->where('doctor_id', '!=', $loggedInDoctorId)
                ->get();

            $doctorIds = $doctorClinicRecords->pluck('doctor_id')->toArray();

            // Step 3: Fetch those doctors
            if (!empty($doctorIds)) {
                $otherDoctors = User::whereIn('id', $doctorIds)
                    ->whereHas('roles', function ($query) {
                        $query->where('name', 'doctor');
                    })
                    ->get();
            }

            // Step 4: Fetch clinic specializations
            $clinicSpecialities = DB::table('clinics')
                ->where('id', $clinicId)
                ->pluck('specialities'); // JSON or CSV

            foreach ($clinicSpecialities as $specs) {
                if (!$specs) continue;

                // JSON format
                $json = json_decode($specs, true);
                if (is_array($json)) {
                    $specializations = $specializations->merge($json);
                } else {
                    // comma-separated string
                    $specializations = $specializations->merge(explode(',', $specs));
                }
            }

            // Unique + sorted
            $specializations = $specializations->unique()->sort()->values();
        }

        // Step 5: Get all patients
        $patients = User::whereHas('roles', function ($query) {
            $query->where('name', 'patient');
        })->get();

        return view('doctor.appointments.appointment-form', compact(
            'loggedInDoctor',
            'otherDoctors',
            'patients',
            'specializations',
            'clinicRecord'
        ));
    }

    // public function getDoctorsBySpecializationName($specialization)
    // {
    //     $doctors = User::where('role', 'doctor')
    //         ->where('specialization', $specialization)
    //         ->get(['id', 'name']);

    //     if ($doctors->isEmpty()) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'No doctors found for this specialization'
    //         ]);
    //     }

    //     return response()->json([
    //         'status' => 'success',
    //         'data' => $doctors
    //     ]);
    // }

    // public function getDoctorsBySpecializationName($specialization)
    // {
    //     $doctors = DB::table('doctors')
    //         ->join('users', 'doctors.user_id', '=', 'users.id')
    //         ->where('doctors.specialization', 'LIKE', '%' . $specialization . '%')
    //         ->whereNull('users.deleted_at')
    //         ->select('users.id', 'users.name')
    //         ->get();

    //     if ($doctors->isEmpty()) {
    //         return response()->json([
    //             'status' => 'error',
    //             'message' => 'No doctors found for this specialization.'
    //         ]);
    //     }

    //     return response()->json([
    //         'status' => 'success',
    //         'data' => $doctors
    //     ]);
    // }

    public function getDoctorsBySpecializationName($specialization)
    {
        $loggedInDoctorId = auth()->user()->id;

        // Step 1: Get clinic(s) of logged-in doctor
        $clinicIds = DB::table('doctor_clinic')
            ->where('doctor_id', $loggedInDoctorId)
            ->pluck('clinic_id');

        if ($clinicIds->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No clinic found for logged-in doctor.'
            ]);
        }

        // Step 2: Get all doctor_ids assigned to these clinic(s)
        $doctorIds = DB::table('doctor_clinic')
            ->whereIn('clinic_id', $clinicIds)
            ->pluck('doctor_id');

        if ($doctorIds->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No doctors found in your clinic(s).'
            ]);
        }

        // Step 3: Filter doctors by specialization
        $doctors = DB::table('doctors')
            ->join('users', 'doctors.user_id', '=', 'users.id')
            ->whereIn('doctors.user_id', $doctorIds)        // only doctors from same clinic
            ->where('doctors.specialization', 'LIKE', '%' . $specialization . '%')
            ->whereNull('users.deleted_at')
            ->select('users.id', 'users.name')
            ->get();

        if ($doctors->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No doctors found for this specialization in your clinic.'
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $doctors
        ]);
    }

    public function getFee($id)
    {
        $availability = DB::table('doctor_availability')
        ->where('doctor_id', $id)
        ->first();

        if ($availability) {
            $timeSlots = json_decode($availability->time_slot, true) ?? [];

            return response()->json([
                'success' => true,
                'slots' => $timeSlots,
                'in_person_fee' => $availability->in_person_fee,
                'video_fee' => $availability->video_fee,
            ]);
        }

        return response()->json(['success' => false]);
    }

    // 
    public function getSpecialization($id)
    {
        $doctor = DB::table('doctors')->where('user_id', $id)->first();

        if (!$doctor || empty($doctor->specialization)) {
            return response()->json([
                'status' => 'error',
                'message' => 'No specialization found',
                'data' => []
            ]);
        }

        $specializations = array_map('trim', explode(',', $doctor->specialization));

        return response()->json([
            'status' => 'success',
            'message' => 'Specialization found',
            'data' => $specializations
        ]);
    }


    // public function getSlots(Request $request, $id)
    // {
    //     $date = $request->query('date');

    //     if (!$date) {
    //         return response()->json(['success' => false, 'message' => 'Date is required']);
    //     }

    //     $dayOfWeek = Carbon::createFromFormat('Y-m-d', $date)->format('l');

    //     $availability = DB::table('doctor_availability')
    //         ->where('doctor_id', $id)
    //         ->where('status', 1)
    //         ->get()
    //         ->filter(function ($item) use ($dayOfWeek) {
    //             $days = json_decode($item->day, true);
    //             return is_array($days) && in_array($dayOfWeek, $days);
    //         });

    //     if ($availability->isEmpty()) {
    //         return response()->json(['success' => false, 'message' => 'Doctor is off on selected date']);
    //     }

    //     $slots = [];
    //     foreach ($availability as $item) {
    //         $times = json_decode($item->time_slot, true);
    //         if (is_array($times)) {
    //             foreach ($times as $slot) {
    //                 $slots[] = $slot;
    //             }
    //         }
    //     }

    //     return response()->json(['success' => true, 'slots' => $slots]);
    // }

    public function getSlots(Request $request, $id)
    {
        $date = $request->query('date');

        if (!$date) {
            return response()->json(['success' => false, 'message' => 'Date is required']);
        }

        $dayOfWeek = Carbon::createFromFormat('Y-m-d', $date)->format('l');

        // Step 1: Get doctor's availability for that day
        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $id)
            ->where('status', 1)
            ->get(['time_slot', 'day']);

        $availableSlots = [];

        foreach ($availability as $item) {
            $days = json_decode($item->day, true);

            if (is_array($days) && in_array($dayOfWeek, $days)) {
                $timeSlots = json_decode($item->time_slot, true);

                if (is_array($timeSlots)) {
                    $availableSlots = array_merge($availableSlots, $timeSlots);
                }
            }
        }

        // if (empty($availableSlots)) {
        //     return response()->json(['success' => false, 'message' => 'Doctor is off on selected date']);
        // }

        // // Step 2: Get already booked slots for that date
        // $bookedSlotsRaw = DB::table('appointments')
        //     ->where('doctor_id', $id)
        //     ->where('appointment_date', $date)
        //     ->pluck('appointment_slot_time')
        //     ->toArray();

        // // Normalize booked slots (just in case there are issues)
        // $bookedSlots = array_map('trim', $bookedSlotsRaw);

        $bookedSlotsRaw = Appointment::where('doctor_id', $id)
        ->whereDate('appointment_date', $date)
        ->pluck('appointment_slot_time')
        ->toArray();

        // Decode all JSON strings inside the array and flatten
        $bookedSlots = [];

        foreach ($bookedSlotsRaw as $jsonString) {
            $slots = json_decode($jsonString, true);
            if (is_array($slots)) {
                $bookedSlots = array_merge($bookedSlots, $slots);
            }
        }

        //dd($availableSlots, $bookedSlots);

        //dd($bookedSlots);
        // Step 3: Match available and booked slots
        $finalSlots = [];

        foreach ($availableSlots as $slot) {
            $slotFormatted = trim($slot);
            $isBooked = in_array($slotFormatted, $bookedSlots);
            $finalSlots[] = [
                'time' => $slotFormatted,
                'status' => $isBooked ? 'booked' : 'available'
            ];
        }

        return response()->json(['success' => true, 'slots' => $finalSlots]);
    }


    public function getAvailability($id)
    {
        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $id)
            ->where('status', 1)
            ->get();

        if ($availability->isEmpty()) {
            return response()->json(['success' => false, 'message' => 'No availability found']);
        }

        $slots = [];
        foreach ($availability as $item) {
            $times = json_decode($item->time_slot, true); // JSON column
            if (is_array($times)) {
                foreach ($times as $slot) {
                    $slots[] = $slot;
                }
            }
        }

        return response()->json(['success' => true, 'slots' => $slots]);
    }

    // public function getAvailability(Request $request, $id)
    // {
    //     $date = $request->query('date'); // Get date from query string

    //     if (!$date) {
    //         return response()->json(['success' => false, 'message' => 'Date is required']);
    //     }

    //     $dayOfWeek = Carbon::createFromFormat('Y-m-d', $date)->format('l'); // e.g., Monday, Tuesday

    //     $availability = DB::table('doctor_availability')
    //         ->where('doctor_id', $id)
    //         ->where('status', 1)
    //         ->where('day', $dayOfWeek) // match day
    //         ->first();

    //     if (!$availability) {
    //         return response()->json(['success' => false, 'message' => 'Doctor is off on selected date']);
    //     }

    //     $slots = [];
    //     $times = json_decode($availability->time_slot, true);
    //     if (is_array($times)) {
    //         foreach ($times as $slot) {
    //             $slots[] = $slot;
    //         }
    //     }

    //     return response()->json([
    //         'success' => true,
    //         'slots' => $slots,
    //         'in_person_fee' => $availability->in_person_fee,
    //         'video_fee' => $availability->video_fee,
    //     ]);
    // }


    public function store(Request $request)
    {
        $validated = $request->validate([
            'patient_id'           => 'required|exists:users,id',
            'doctor_id'            => 'required|exists:users,id',
            'clinic_id'            => 'required|integer',
            'specialization_name'  => 'required|string',
            //'appointment_slot_time'=> 'required', // assuming it's a JSON array (string)
            'appointment_slot_time' => 'required|string', // get as string, will be converted
            'total_price'          => 'required|numeric',
            'appointment_type'     => 'required|string|in:first_visit,follow_up',
            'appointment_status'   => 'required|string|in:cancelled,online,offline',
            'fee_type'             => 'required|string|in:in_person_fee,video_fee', // add allowed values accordingly
            'appointment_status'   => 'required|string|in:cancelled,online,offline',
            'reason'               => 'nullable|string|max:1000',
        ]);

        // ✅ Auto-generate appointment_id like APT2025001
        $latest = \App\Models\Appointment::latest()->first();
        $number = $latest && $latest->appointment_id
            ? ((int) substr($latest->appointment_id, 3)) + 1
            : 1;
        $appointment_id = 'APT' . str_pad($number, 5, '0', STR_PAD_LEFT); // APT0000001

        // Convert slot string to array
        $slotTimes = array_map('trim', explode(',', $validated['appointment_slot_time']));

        // ✅ Insert data
        \App\Models\Appointment::create([
            'appointment_id'        => $appointment_id,
            'patient_id'            => $validated['patient_id'],
            'doctor_id'             => $validated['doctor_id'],
            'clinic_id'             => $validated['clinic_id'],
            'specialization'        => $validated['specialization_name'],
            'appointment_date'      => now()->format('Y-m-d'), // or pass separately if needed
            //'appointment_slot_time' => json_encode($validated['appointment_slot_time']),
            'appointment_slot_time' => json_encode($slotTimes),
            'appointment_fee'       => $validated['total_price'],
            'fee_type'              => $validated['fee_type'], // optional, set as needed  
            'appointment_type'      => $validated['appointment_type'],
            'appointment_status'    => $validated['appointment_status'],
            'reason'                => $validated['reason'],
        ]);

        return redirect()->back()->with('success', 'Appointment created successfully!');
    }

    public function destroy($id)
    {
        $appointment = Appointment::find($id);

        if (!$appointment) {
            return redirect()->back()->with('error', 'Appointment not found.');
        }

        $appointment->delete();

        return redirect()->back()->with('success', 'Appointment deleted successfully.');
    }

        // public function show($id)
    // {
    //     $appointment = Appointment::with(['patient', 'doctor', 'clinic']) // assuming clinic relation
    //         ->findOrFail($id);

    //     return response()->json([
    //         'appointment_id' => $appointment->appointment_id,
    //         'appointment_date' => $appointment->appointment_date,
    //         'appointment_slot_time' => json_decode($appointment->appointment_slot_time),
    //         'appointment_type' => $appointment->appointment_type,
    //         'clinic_name' => $appointment->clinic->name ?? 'Unknown',
    //         'patient_name' => $appointment->patient->name ?? 'Unknown',
    //         'patient_image' => asset('path/to/default-avatar.jpg'), // Optional: add real image path
    //     ]);
    // }

    public function show($id)
    {
        $appointment = Appointment::with(['doctor', 'patient']) // make sure relationships exist
            ->findOrFail($id);

        return response()->json($appointment);
    }

    public function updateStatus($id, $status)
    {
        $allowed = ['completed', 'cancelled'];

        if (!in_array($status, $allowed)) {
            return redirect()->back()->withErrors('Invalid status');
        }

        $appointment = Appointment::findOrFail($id);

        if ($appointment->appointment_status == $status) {
            return redirect()->back()->with('info', "Appointment already marked as $status.");
        }

        $appointment->appointment_status = $status;
        $appointment->save();

        return redirect()->back()->with('success', "Appointment status updated to $status.");
}


    // public function edit($id)
    // {
    //     $appointment = Appointment::findOrFail($id);
    //     $patients = User::where('role', 'patient')->get();

    //     return view('doctor.appointments.appointment-form', compact('appointment', 'patients'));
    // }

    // public function view($id)
    // {
    //     $appointment = Appointment::findOrFail($id);
    //     return view('doctor.appointments.view', compact('appointment'));
    // }

    public function createNewPatient(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:255',
            'email'    => 'required|email|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'status'   => 'required|string',
            'phone'    => 'required|string',
            'date_of_birth' => 'required|date',
            'gender'   => 'required|string',
            'address'  => 'required|string',
            'profile_image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        try {

            //dd($request->all());
            // Create user
            $user = User::create([
                'name'     => $request->name,
                'email'    => $request->email,
                'password' => Hash::make($request->password),
                'status'   => $request->status,
            ]);

            // Assign role
            $patientRole = Role::where('name', 'patient')->where('guard_name', 'web')->firstOrFail();
            $user->assignRole($patientRole);

            // Handle profile image
            $profileImagePath = null;
            if ($request->hasFile('profile_image')) {
                $profileImagePath = $request->file('profile_image')->store('patients', 'public');
            }

            // Create patient record
            $patient = Patient::create([
                'user_id'         => $user->id,
                'address'         => $request->address,
                'phone'           => $request->phone,
                'date_of_birth'   => $request->date_of_birth,
                'gender'          => $request->gender,
                'medical_history' => $request->medical_history,
                'profile_image'   => $profileImagePath,
            ]);

            // 3️⃣ Create patient_clinic record
            DB::table('patient_clinics')->insert([
                'patient_id'      => $user->id,
                'clinic_id'       => $request->clinic_id, // Or however you get the clinic ID
                'doctor_id'       => $request->doctor_id,
                'added_by'        => Auth::id(),
                'status'          => 'active',
                'approval_status' => 'approved',
                'approved_at'     => now(),
                'notes'           => json_encode($request->notes ?? []),
                'created_at'      => now(),
                'updated_at'      => now(),
            ]);
            
            // Associate doctor & clinic if provided
            // if ($request->selectedDoctor) {
            //     // createPatientClinicAssociation($patient) logic here
            // }
            return redirect()->back()->with('success', 'Patient created successfully.');
            //return redirect()->route('doctor.appointments.appointment-form')->with('success', 'Patient created successfully.');
        } catch (\Exception $e) {
            Log::error('Patient creation failed: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to create patient: ' . $e->getMessage()]);
        }
    }
}
