<?php 

namespace App\Http\Controllers\Doctor\Api\V1;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

use Carbon\Carbon;

use App\Models\DoctorClinic;  
use App\Models\Clinic;
use App\Models\Appointment;

class DoctorClinicApiController extends Controller
{
    // Fetch Clinic List 
    public function getClinics(Request $request)
    {
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id',
        ]);

        // ✅ doctor_id ke hisaab se clinic_id list
        $clinicIds = DoctorClinic::where('doctor_id', $request->doctor_id)
            ->pluck('clinic_id');

        if ($clinicIds->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No clinics found for this doctor.',
                'data'   => [],
            ], 404);
        }

        // ✅ Clinic table se full details nikal lo
        $clinics = Clinic::whereIn('id', $clinicIds)
            ->select('id', 'name')
            ->get();
        
        //$clinics = Clinic::whereIn('id', $clinicIds)
        //    ->select('id', 'name', 'line1', 'line2', 'city', 'state', 'country', 'pincode', 'contact_numbers', 'status')
        //    ->get();

        return response()->json([
            'status'  => 'success',
            'message' => 'Clinics fetched successfully.',
            'data'    => $clinics,
        ], 200);
    }

    // Doctor Add Availability
    public function addAvailability(Request $request)
    {
        $validated = $request->validate([
            'clinic_id'     => 'required|integer',
            'doctor_id'     => 'required|integer|exists:users,id',
            'slot_duration' => 'required|integer|min:1',
            'from_time'     => 'required|date_format:H:i',
            'to_time'       => 'required|date_format:H:i|after:from_time',
            'in_person_fee' => 'required|numeric|min:0',
            'video_fee'     => 'required|numeric|min:0',
            'days'          => 'required|string', // now string
        ]);

        // Convert comma-separated string to array
        $daysArray = array_map('trim', explode(',', $validated['days']));

        if (empty($daysArray)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Days parameter cannot be empty.'
            ], 422);
        }

        $doctorId = $validated['doctor_id'];
        $clinicId = $validated['clinic_id'];

        $from = Carbon::createFromFormat('H:i', $validated['from_time']);
        $to   = Carbon::createFromFormat('H:i', $validated['to_time']);
        $step = (int) $validated['slot_duration'];

        $slots = [];
        while ($from->copy()->addMinutes($step)->lte($to)) {
            $slotStart = $from->format('H:i');
            $slotEnd   = $from->copy()->addMinutes($step)->format('H:i');
            $slots[]   = "$slotStart - $slotEnd";
            $from->addMinutes($step);
        }

        // Check if doctor already has availability in this clinic
        $exists = DB::table('doctor_availability')
            ->where('doctor_id', $doctorId)
            ->where('clinic_id', $clinicId)
            ->exists();

        if ($exists) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Availability for this doctor in this clinic already exists.'
            ], 422);
        }

        // ✅ Insert availability
        DB::table('doctor_availability')->insert([
            'clinic_id'         => $clinicId,
            'doctor_id'         => $doctorId,
            'slot_duration'     => $validated['slot_duration'],
            'day'               => json_encode($daysArray), // store as JSON array
            'from_time'         => $validated['from_time'],
            'to_time'           => $validated['to_time'],
            'time_slot'         => json_encode($slots),
            'booked_time_slot'  => json_encode([]),
            'in_person_fee'     => $validated['in_person_fee'],
            'video_fee'         => $validated['video_fee'],
            'status'            => '1',
            'other'             => '',
            'created_at'        => now(),
            'updated_at'        => now(),
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Availability created successfully.',
            'data'    => [
                'from_time' => $validated['from_time'],
                'to_time' => $validated['to_time'],
                'slots' => $slots,
                'days'  => $daysArray,
            ]
        ], 200);
    }

    // Doctor View Availability
    public function getAvailability(Request $request)
    {
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id',
            'clinic_id' => 'required|integer',
        ]);

        $availability = DB::table('doctor_availability')
            ->where('doctor_id', $request->doctor_id)
            ->where('clinic_id', $request->clinic_id)
            ->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'No availability found for this doctor in this clinic.'
            ], 404);
        }

        return response()->json([
            'status' => 'success',
            'data' => [
                'availability_id'  => $availability->id,
                'clinic_id'        => $availability->clinic_id,
                'doctor_id'        => $availability->doctor_id,
                'slot_duration'    => $availability->slot_duration,
                'from_time'        => $availability->from_time,
                'to_time'          => $availability->to_time,
                'days'             => json_decode($availability->day, true),
                'slots'            => json_decode($availability->time_slot, true),
                //'booked_slots'     => json_decode($availability->booked_time_slot, true),
                'in_person_fee'    => $availability->in_person_fee,
                'video_fee'        => $availability->video_fee,
                'status'           => $availability->status,
                //'other'            => $availability->other,
            ]
        ], 200);
    }

    // Doctor Update Availability
    public function updateAvailability(Request $request)
    {
        $validated = $request->validate([
            'id'            => 'required|integer|exists:doctor_availability,id',
            'slot_duration' => 'sometimes|integer|min:1',
            'from_time'     => 'sometimes|date_format:H:i',
            'to_time'       => 'sometimes|date_format:H:i|after:from_time',
            'in_person_fee' => 'sometimes|numeric|min:0',
            'video_fee'     => 'sometimes|numeric|min:0',
            'days'          => 'sometimes|string', // comma-separated
        ]);

        $availability = DB::table('doctor_availability')->where('id', $validated['id'])->first();

        if (!$availability) {
            return response()->json([
                'status' => 'error',
                'message' => 'Availability not found.'
            ], 404);
        }

        $updateData = [];

        if (isset($validated['slot_duration'])) $updateData['slot_duration'] = $validated['slot_duration'];
        if (isset($validated['from_time'])) $updateData['from_time'] = $validated['from_time'];
        if (isset($validated['to_time'])) $updateData['to_time'] = $validated['to_time'];
        if (isset($validated['in_person_fee'])) $updateData['in_person_fee'] = $validated['in_person_fee'];
        if (isset($validated['video_fee'])) $updateData['video_fee'] = $validated['video_fee'];
        if (isset($validated['days'])) {
            $daysArray = array_map('trim', explode(',', $validated['days']));
            $updateData['day'] = json_encode($daysArray);
        }

        // ✅ Recalculate slots if from_time, to_time or slot_duration changed
        if (isset($validated['from_time']) || isset($validated['to_time']) || isset($validated['slot_duration'])) {
            $from = isset($validated['from_time'])
                ? Carbon::createFromFormat('H:i', $validated['from_time'])
                : Carbon::createFromFormat('H:i', $availability->from_time);

            $to = isset($validated['to_time'])
                ? Carbon::createFromFormat('H:i', $validated['to_time'])
                : Carbon::createFromFormat('H:i', $availability->to_time);

            $step = isset($validated['slot_duration'])
                ? (int) $validated['slot_duration']
                : $availability->slot_duration;

            $slots = [];
            while ($from->copy()->addMinutes($step)->lte($to)) {
                $slotStart = $from->format('H:i');
                $slotEnd   = $from->copy()->addMinutes($step)->format('H:i');
                $slots[]   = "$slotStart - $slotEnd";
                $from->addMinutes($step);
            }

            $updateData['time_slot'] = json_encode($slots);
        }

        $updateData['updated_at'] = now();

        DB::table('doctor_availability')->where('id', $validated['id'])->update($updateData);

        return response()->json([
            'status' => 'success',
            'message' => 'Availability updated successfully.',
            'data' => $updateData
        ], 200);
    }

    // Doctor Appoinment List
    public function getDoctorAppointments(Request $request)
    {
        // ✅ Validate doctor_id
        $validated = $request->validate([
            'doctor_id' => 'required|exists:users,id',
            'appointment_status' => 'nullable|string|in:cancelled,online,offline,completed,pending,confirmed'
        ]);

        $doctorId = $validated['doctor_id'];
        $status   = $validated['appointment_status'] ?? null;

        // ✅ Build Query with Patient Relation
        $query = Appointment::with(['patient:id,name,email'])
            ->where('doctor_id', $doctorId);

        if ($status) {
            // ✅ If status is given, apply it directly
            $query->where('appointment_status', $status);
        } else {
            // ✅ Otherwise exclude completed and cancelled
            $query->whereNotIn('appointment_status', ['completed', 'cancelled','pending','confirmed']);
        }

        $appointments = $query->orderBy('id', 'DESC')->get();

        // ✅ Merge patient details into each appointment
        $appointments->transform(function ($appointment) {
            $appointment->patient_name  = $appointment->patient->name ?? null;
            $appointment->patient_email = $appointment->patient->email ?? null;
            unset($appointment->patient); // remove nested patient object
            return $appointment;
        });

        return response()->json(
            // 'status'  => 'success',
            // 'message' => $appointments->isEmpty() ? 'No appointments found' : 'Appointments fetched successfully',
            $appointments
        , 200);
    }

    // Doctor Accept Appoinment List
    public function acceptAppoinments(Request $request)
    {
        $validated = $request->validate([
            'appointment_id'     => 'required|integer|exists:appointments,id',
            'appointment_status' => 'required|string|in:confirmed', // sirf ye 2 allow
            'reason'             => 'nullable|string|max:255', // optional, sirf cancel pe use kare
        ]);

        $appointment = DB::table('appointments')->where('id', $validated['appointment_id'])->first();

        if (!$appointment) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Appointment not found.'
            ], 404);
        }

        // ✅ Update appointment status
        DB::table('appointments')
            ->where('id', $validated['appointment_id'])
            ->update([
                'appointment_status' => $validated['appointment_status'],
                'reason'             => $validated['reason'] ?? $appointment->reason,
                'updated_at'         => now(),
            ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Appointment status updated successfully.',
            'appointment_id'     => $validated['appointment_id'],
            'appointment_status' => $validated['appointment_status'],
            //'reason'             => $validated['reason'] ?? $appointment->reason,
        ], 200);
    }

    // Doctor Update Appoinment List 
    public function updateAppointmentStatus(Request $request)
    {
        $validated = $request->validate([
            'appointment_id'     => 'required|integer|exists:appointments,id',
            'appointment_status' => 'required|string|in:cancelled,completed', // sirf ye 2 allow
            'reason'             => 'nullable|string|max:255', // optional, sirf cancel pe use kare
        ]);

        $appointment = DB::table('appointments')->where('id', $validated['appointment_id'])->first();

        if (!$appointment) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Appointment not found.'
            ], 404);
        }

        // ✅ Update appointment status
        DB::table('appointments')
            ->where('id', $validated['appointment_id'])
            ->update([
                'appointment_status' => $validated['appointment_status'],
                'reason'             => $validated['reason'] ?? $appointment->reason,
                'updated_at'         => now(),
            ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Appointment status updated successfully.',
            'appointment_id'     => $validated['appointment_id'],
            'appointment_status' => $validated['appointment_status'],
            //'reason'             => $validated['reason'] ?? $appointment->reason,
        ], 200);
    }

}
