<?php

namespace App\Http\Controllers\Doctor\Api\V1;
use Laravel\Sanctum\PersonalAccessToken;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;  
use Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage; // ✅ Yahi missing tha
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;

use App\Models\ActiveDevice;
use App\Models\Appointment;
use App\Models\Doctors;
use App\Models\Doctor;
use App\Models\User; // Assuming Patient is stored in users table with 'patient' role

class DoctorApiAuthController extends Controller
{
    // Send OTP 
    public function sendOtp(Request $request)     
    {
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
        ]);

        if (!$request->email && !$request->phone_number) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or phone number is required.'
            ], 422);
        }

        // ✅ Check if doctor exists based on email
        if ($request->email) {
            $user = User::where('email', $request->email)
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
                })
                ->first();

            if ($user) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'This email is already registered with a doctor.'
                ], 400);
            }
        }

        // ✅ Check if doctor exists based on phone number
        if ($request->phone_number) {
            $user = User::where('phone_number', $request->phone_number)
                ->whereHas('roles', function ($q) {
                    $q->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
                })
                ->first();

            if ($user) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'This phone number is already registered with a doctor.'
                ], 400);
            }
        }

        // ✅ Static OTP for now
        $otp = '1234';

        // Normally: Save OTP in DB or cache for verification

        return response()->json([
            'status'  => 'success',
            'message' => 'OTP sent successfully for doctor registration.',
            'otp'     => $otp // ✅ For testing only
        ]);
    }

    // Verify OTP
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
            'otp'          => 'required|string'
        ]);

        if (!$request->email && !$request->phone_number) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or phone number is required.'
            ], 422);
        }

        // ✅ Static OTP check
        if ($request->otp !== '1234') {
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid OTP.'
            ], 400);
        }

        // ✅ OTP sahi hai, doctor exist kare ya na kare → success
        return response()->json([
            'status'  => 'success',
            'message' => 'OTP verified successfully.'
        ]);
    }

    // Signup 
    // public function signup(Request $request)
    // {
    //     $messages = [
    //         'email.unique'        => 'This email is already registered.',
    //         'phone_number.unique' => 'This phone number is already registered.',
    //     ];

    //     $validator = Validator::make($request->all(), [
    //         'name'             => 'required|string|max:255',
    //         'email'            => 'required|email|unique:users,email',
    //         'phone_number'     => 'required|string|min:10|max:15|unique:users,phone_number',
    //         'specialization'   => 'required|string',
    //         'license_number'   => 'required|string|unique:doctors,license_number',
    //         'qualifications'   => 'required|string',
    //         'experience_years' => 'required|integer|min:0',
    //         'bio'              => 'nullable|string',
    //     ], $messages);

    //     if ($validator->fails()) {
    //         return response()->json([
    //             'status'  => 'error',
    //             'message' => $validator->errors()->first(),
    //         ], 422);
    //     }

    //     DB::beginTransaction();
    //     try {
    //         $specializations = array_map('trim', explode(',', $request->specialization));
    //         $specializationsJson = json_encode($specializations);

    //         // ✅ Step 1: Create user
    //         $user = User::create([
    //             'name'              => $request->name,
    //             'email'             => $request->email,
    //             'phone_number'      => $request->phone_number,
    //             'status'            => 'inactive',
    //             'email_verified_at' => now(),
    //         ]);

    //         $user->assignRole('doctor');

    //         // ✅ Step 2: Create doctor profile
    //         $doctor = Doctor::create([
    //             'user_id'         => $user->id,
    //             'status'          => 'inactive',
    //             'specialization'  => $specializationsJson,
    //             'license_number'  => $request->license_number,
    //             'qualifications'  => $request->qualifications,
    //             'experience_years'=> $request->experience_years,
    //             'bio'             => $request->bio,
    //         ]);

    //         // ✅ Step 3: Generate Bearer Token (Laravel Sanctum or Passport)
    //         $token = $user->createToken('DoctorAPI')->plainTextToken;

    //         DB::commit();

    //         return response()->json([
    //             'status'  => 'success',
    //             'message' => 'Doctor registered successfully.',
    //             'access_token' => $token,
    //             'token_type'   => 'Bearer',
    //             'doctor_id'       => $doctor->id,
    //             //'name'            => $user->name,
    //             //'email'           => $user->email,
    //             //'phone_number'    => $user->phone_number,
    //             //'status'          => $doctor->status,
    //             //'specialization'  => $doctor->specialization,
    //             //'license_number'  => $doctor->license_number,
    //             //'qualifications'  => $doctor->qualifications,
    //             //'experience_years'=> $doctor->experience_years,
    //             //'bio'             => $doctor->bio,
    //             //'created_at'      => $doctor->created_at,
    //         ], 200);

    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         Log::error('Doctor signup failed', ['error' => $e->getMessage()]);
    //         return response()->json([
    //             'status'  => 'error',
    //             'message' => 'Doctor registration failed.',
    //             'error'   => $e->getMessage(),
    //         ], 500);
    //     }
    // }

    public function signup(Request $request)
    {
        $messages = [
            'email.unique'        => 'This email is already registered.',
            'phone_number.unique' => 'This phone number is already registered.',
        ];

        $validator = Validator::make($request->all(), [
            'name'             => 'required|string|max:255',
            'email'            => 'required|email|unique:users,email',
            'phone_number'     => 'required|string|min:10|max:15|unique:users,phone_number',
            'specialization'   => 'required|string',
            'license_number'   => 'required|string|unique:doctors,license_number',
            'qualifications'   => 'required|string',
            'experience_years' => 'required|integer|min:0',
            'bio'              => 'nullable|string',
            'certificate.*'    => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048', // multiple files allowed
        ], $messages);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => $validator->errors()->first(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            $specializations = array_map('trim', explode(',', $request->specialization));
            $specializationsJson = json_encode($specializations);

            // ✅ Step 1: Create user
            $user = User::create([
                'name'              => $request->name,
                'email'             => $request->email,
                'phone_number'      => $request->phone_number,
                'status'            => 'inactive',
                'email_verified_at' => now(),
            ]);

            $user->assignRole('doctor');

            // ✅ Step 2: Handle certificate uploads
            $certificatePaths = []; // always initialize
            if ($request->hasFile('certificate')) {
                foreach ($request->file('certificate') as $file) {
                    $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs('certificates', $filename, 'public');
                    $certificatePaths[] = $path;
                }
            }

            // ✅ Step 3: Create doctor profile
            $doctor = Doctor::create([
                'user_id'          => $user->id,
                'status'           => 'inactive',
                'specialization'   => $specializationsJson,
                'license_number'   => $request->license_number,
                'qualifications'   => $request->qualifications,
                'experience_years' => $request->experience_years,
                'bio'              => $request->bio,
                'certificate'      => !empty($certificatePaths) ? json_encode($certificatePaths) : null,
            ]);

            // ✅ Step 4: Generate Bearer Token
            $token = $user->createToken('DoctorAPI')->plainTextToken;

            DB::commit();

            return response()->json([
                'status'       => 'success',
                'message'      => 'Doctor registered successfully.',
                'access_token' => $token,
                'token_type'   => 'Bearer',
                'doctor_id'    => $doctor->id,
                'certificate'  => $certificatePaths, // return uploaded paths
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Doctor signup failed', ['error' => $e->getMessage()]);
            return response()->json([
                'status'  => 'error',
                'message' => 'Doctor registration failed.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * ✅ Send OTP for Doctor Login
     */
    public function loginOtp(Request $request)
    {
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
        ]);

        if (!$request->email && !$request->phone_number) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or phone number is required.'
            ], 422);
        }

        // ✅ Find doctor user based on email OR phone with role check
        $query = User::query()->whereHas('roles', function ($q) {
            $q->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
        });

        if ($request->email) {
            $query->where('email', $request->email);
        } else {
            $query->where('phone_number', $request->phone_number);
        }

        $user = $query->first();

        if (!$user) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Doctor not registered. Please signup first.'
            ], 404);
        }

        // ✅ Check if user is approved (status == active)
        if ($user->status !== 'active') {
            return response()->json([
                'status'  => 'error',
                'message' => 'Your account has not been approved by the admin yet. Please wait until it is approved before trying again.'
            ], 403);
        }

        // ✅ Generate OTP (static for now)
        $otp = '1234';

        // ⚡ Normally save OTP to DB or cache with expiry
        // Otp::create([ 'user_id' => $user->id, 'otp' => $otp, 'expires_at' => now()->addMinutes(5) ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'OTP sent successfully for doctor login.',
            'otp'     => $otp // ✅ Only for testing, remove in production
        ], 200);
    }

    /**
     * ✅ Doctor Login with OTP
     */
    public function login(Request $request)
    {
        // ✅ Validate केवल email या phone के लिए
        $request->validate([
            'email'        => 'nullable|email',
            'phone_number' => 'nullable|string|min:10|max:15',
            'otp'          => 'required|string|min:4|max:6',
        ]);

        // ✅ OTP Check
        if ($request->otp !== '1234') {
            return response()->json([
                'status'  => 'error',
                'message' => 'Invalid OTP.'
            ], 400);
        }

        // ✅ Check email या phone से user find करो (doctor only)
        if (!empty($request->email)) {
            $query = User::where('email', $request->email);
        } elseif (!empty($request->phone_number)) {
            $query = User::where('phone_number', $request->phone_number);
        } else {
            return response()->json([
                'status'  => 'error',
                'message' => 'Email or Phone number is required.'
            ], 422);
        }

        $user = $query->whereHas('roles', function ($q) {
            $q->whereIn('name', ['doctor', 'primary_doctor', 'sub_doctor']);
        })->first();

        if (!$user) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Doctor not found. Please register first.'
            ], 404);
        }

        // ✅ Static device details (later: pick from request)
        $deviceName = 'DoctorApp';
        $deviceId   = '12345';

        // ✅ Create token
        $token = $user->createToken($deviceName)->plainTextToken;

        // ✅ Track Device (optional)
        // $this->trackDevice($user, $request, $token, $deviceId);

        return response()->json([
            'status'  => 'success',
            'message' => 'Doctor logged in successfully',
            'access_token' => $token,
            'token_type'   => 'Bearer',
            'doctor_id'      => $user->id,
            'role'         => $user->roles->pluck('name')->first(), // 👈 doctor role bhi return hoga
            // 'data' => [
            //     'user' => $this->getUserResponse($user),
            // ]
        ]);      
    }

    // Logout 
    public function logout(Request $request)
    {
        try {
            $request->validate([
                'doctor_id' => 'nullable|integer|exists:users,id',
            ]);

            // ✅ CASE 1: Agar doctor_id diya ho (force logout all sessions)
            if ($request->filled('doctor_id')) {
                $user = User::find($request->doctor_id);

                if (!$user) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Doctor not found',
                    ], 404);
                }

                // ✅ Saare tokens delete kardo
                PersonalAccessToken::where('tokenable_id', $user->id)
                    ->where('tokenable_type', User::class)
                    ->delete();

                // ✅ ActiveDevice table bhi update (agar use kar rahe ho)
                // ActiveDevice::where('user_id', $user->id)->update([
                //     'is_active' => false,
                //     'logout_at' => now(),
                //     'fcm_token' => null,
                // ]);

                return response()->json([
                    'status' => 'success',
                    'message' => 'Doctor logged out from all devices',
                ]);
            }

            // ✅ CASE 2: Agar token diya ho (normal logout)
            $user = $request->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Unauthorized',
                ], 401);
            }

            $token = $user->currentAccessToken();
            if (!$token) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No active session found',
                ], 400);
            }

            $token->delete();

            // ✅ ActiveDevice table update (agar use kar rahe ho)
            // ActiveDevice::where('user_id', $user->id)
            //     ->where('token_id', $token->id)
            //     ->update([
            //         'is_active' => false,
            //         'logout_at' => now(),
            //         'fcm_token' => null,
            //     ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Successfully logged out',
            ]);

        } catch (\Exception $e) {
            Log::error('Doctor Logout error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to logout',
            ], 500);
        }
    }

    // Doctor Profile Details 
    public function getProfile(Request $request)
    {
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id', // ✅ Ye user table ka ID hai
        ]);

        // ✅ Fetch doctor where user_id = doctor_id
        $doctor = Doctor::with('user')->where('user_id', $request->doctor_id)->first();

        if (!$doctor || !$doctor->user || !$doctor->user->hasRole(['doctor','primary_doctor','sub_doctor'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'Doctor not found or not authorized.'
            ], 404);
        }

        return response()->json([
            //'status'  => 'success',
            'message' => 'Doctor profile fetched successfully.',

            'doctor_id'       => $doctor->user_id, // doctor table ka id
            // ✅ User table fields
            'name'            => $doctor->user->name,
            'email'           => $doctor->user->email,
            'phone_number'    => $doctor->user->phone_number,
            //'user_status'     => $doctor->user->status,

            // ✅ Doctor table fields
            'booking_type'    => $doctor->booking_type,
            'specialization'  => $doctor->specialization ? json_decode($doctor->specialization, true) : [],
            'license_number'  => $doctor->license_number,
            'qualifications'  => $doctor->qualifications,
            'experience_years'=> $doctor->experience_years,
            'bio'             => $doctor->bio,
            'profile_image'   => $doctor->profile_image ? asset('storage/'.$doctor->profile_image) : null,
            // ✅ Correct multiple certificates output
            'certificates'    => $doctor->certificate 
                                    ? collect(json_decode($doctor->certificate, true))
                                        ->map(fn($path) => asset('storage/'.$path))
                                        ->toArray()
                                    : [],
            'status'          => $doctor->status,

        ], 200);
    }

    // Update Doctor Profile Details 
    public function updateProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'doctor_id'        => 'required|integer|exists:users,id',

            // ✅ User Table Fields
            'name'             => 'sometimes|string|max:255',
            'email'            => 'sometimes|email|unique:users,email,' . $request->doctor_id,
            'phone_number'     => 'sometimes|string|min:10|max:15|unique:users,phone_number,' . $request->doctor_id,

            // ✅ Doctor Table Fields
            'specialization'   => 'sometimes|string',
            'license_number'   => 'sometimes|string',
            'qualifications'   => 'sometimes|string',
            'experience_years' => 'sometimes|integer|min:0',
            'bio'              => 'nullable|string',
            'profile_image'    => 'nullable|image|mimes:jpeg,png,jpg|max:2048',

            // ✅ Booking Type
            'booking_type'     => 'sometimes|in:automatic,manual',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => $validator->errors()->first(),
            ], 422);
        }

        DB::beginTransaction();
        try {
            $user = User::findOrFail($request->doctor_id);
            $user->update($request->only(['name', 'email', 'phone_number']));

            $doctor = Doctor::where('user_id', $user->id)->first();
            if (!$doctor) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Doctor profile not found.'
                ], 404);
            }

            // ✅ Convert specialization string → JSON array
            $specializationsJson = $doctor->specialization;
            if ($request->filled('specialization')) {
                $specializationsArray = array_map('trim', explode(',', $request->specialization));
                $specializationsJson = json_encode($specializationsArray);
            }

            $updateData = [
                'specialization'   => $specializationsJson,
                'license_number'   => $request->license_number ?? $doctor->license_number,
                'qualifications'   => $request->qualifications ?? $doctor->qualifications,
                'experience_years' => $request->experience_years ?? $doctor->experience_years,
                'bio'              => $request->bio ?? $doctor->bio,
                'booking_type'     => $request->filled('booking_type') ? $request->booking_type : $doctor->booking_type,
            ];

            // ✅ Handle Profile Image
            if ($request->hasFile('profile_image')) {
                if ($doctor->profile_image && Storage::disk('public')->exists($doctor->profile_image)) {
                    Storage::disk('public')->delete($doctor->profile_image);
                }

                $path = $request->file('profile_image')->store('doctor-profiles', 'public');
                $updateData['profile_image'] = $path;
            }

            $doctor->update($updateData);

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Profile updated successfully.',
                'data'    => [
                    'doctor_id'        => $doctor->id,
                    'name'             => $user->name,
                    'email'            => $user->email,
                    'phone_number'     => $user->phone_number,
                    'specialization'   => json_decode($doctor->specialization, true),
                    'license_number'   => $doctor->license_number,
                    'qualifications'   => $doctor->qualifications,
                    'experience_years' => $doctor->experience_years,
                    'bio'              => $doctor->bio,
                    'profile_image'    => $doctor->profile_image ? asset('storage/'.$doctor->profile_image) : null,
                    'booking_type'     => $doctor->booking_type,
                ]
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Doctor profile update failed', ['error' => $e->getMessage()]);
            return response()->json([
                'status'  => 'error',
                'message' => 'Profile update failed.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // Dashboard All Data 
    public function dashboard(Request $request)
    {
        $request->validate([
            'doctor_id' => 'required|integer|exists:users,id',
        ]);

        $doctorId = $request->doctor_id;

        // ✅ Total Appointments
        $totalAppointments = DB::table('appointments')
            ->where('doctor_id', $doctorId)
            ->count();

        // ✅ Completed Appointments (for earning calculation)
        $completedAppointments = DB::table('appointments')
            ->where('doctor_id', $doctorId)
            ->where('appointment_status', 'completed')
            ->get();

        $totalEarnings = $completedAppointments->sum('appointment_fee');

        // ✅ Teleconsultation Count (video_fee wale appointments)
        $teleconsultations = DB::table('appointments')
            ->where('doctor_id', $doctorId)
            ->where('fee_type', 'video_fee')
            ->count();

        return response()->json([
            'status'  => 'success',
            'message' => 'Dashboard data fetched successfully.',
            'data'    => [
                'total_appointments'   => $totalAppointments,
                'completed_appointments' => $completedAppointments->count(),
                'total_earnings'       => (float) $totalEarnings,
                'teleconsultations'    => $teleconsultations,
            ]
        ], 200);
    }

}
