<?php

namespace App\Http\Controllers\Doctor\Api\V1;

use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;  
use Illuminate\Support\Facades\DB;

use Illuminate\Support\Facades\Log;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

use App\Models\Appointment; // Model use karenge

class CommonApiController extends Controller
{
    /**
     * ✅ Fetch all specializations
     */
    public function getSpecializations()
    {
        try {
            $specializations = \DB::table('specialization')
                ->select('id', 'specialization as name')
                ->orderBy('specialization', 'ASC')
                ->get();

            return response()->json([
                'status'  => 'success',
                'message' => 'Specializations fetched successfully.',
                'data'    => $specializations,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong!',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * ✅ Fetch all clinic
     */
    public function getClinics()
    {
        try {
            $clinics = \DB::table('clinics')
                ->select('id', 'name')
                ->whereNull('deleted_at') // ✅ Soft delete filter
                ->orderBy('name', 'ASC')
                ->get();

            return response()->json([
                'status'  => 'success',
                'message' => 'Clinics fetched successfully.',
                'data'    => $clinics,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Something went wrong!',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * ✅ Update meeting id and token 
     */
    public function updateMeetingId(Request $request)
    {
        $apiKey = env('VIDEOSDK_API_KEY');
        $secret = env('VIDEOSDK_SECRET_KEY');

        if (empty($apiKey) || empty($secret)) {
            Log::error('VideoSDK credentials missing');
            return response()->json([
                'error' => 'Server config error',
                'message' => 'Missing API key or secret'
            ], 500);
        }

        try {
            // ✅ JWT Token generate
            $payload = [
                'apikey' => $apiKey,
                'permissions' => ['allow_join', 'allow_mod'],
                'version' => 2,
                'iat' => time(),
                //'exp' => time() + 3600,
                //'exp' => time() + (60 * 60 * 24 * 7),
                'exp' => time() + 604800,  // ✅ 7 din ka expiry
            ];

            $token = JWT::encode($payload, $secret, 'HS256');
        } catch (\Exception $e) {
            Log::error('JWT generation failed', [
                'message' => $e->getMessage()
            ]);
            return response()->json([
                'error' => 'Token generation error',
                'message' => $e->getMessage()
            ], 500);
        }

        // ✅ Request validation
        $request->validate([
            'appointment_id' => 'required|integer',
            'doctor_id'      => 'required|integer',
            'patient_id'     => 'required|integer',
            'meeting_id'     => 'required|string|max:255',
        ]);

        // ✅ Appointment record find karo
        $appointment = Appointment::where('id', $request->appointment_id)
            ->where('doctor_id', $request->doctor_id)
            ->where('patient_id', $request->patient_id)
            ->first();

        if (!$appointment) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Appointment not found',
            ], 404);
        }

        // ✅ meeting_id aur token update karo
        $appointment->meeting_id = $request->meeting_id;
        $appointment->token = $token;
        $appointment->save();

        return response()->json([
            'status'  => 'success',
            'message' => 'Meeting ID and token updated successfully',
            'data'    => [
                'appointment_id' => $appointment->id,
                'meeting_id'     => $appointment->meeting_id,
                'token'          => $appointment->token,
            ],
        ], 200);
    }

    // New APi with Meeting ID 
    public function updateMeetingIdToken(Request $request)
    {
        $apiKey = env('VIDEOSDK_API_KEY');
        $secret = env('VIDEOSDK_SECRET_KEY');

        if (empty($apiKey) || empty($secret)) {
            Log::error('VideoSDK credentials missing');
            return response()->json([
                'error' => 'Server config error',
                'message' => 'Missing API key or secret'
            ], 500);
        }

        try {
            // ✅ JWT Token generate
            $payload = [
                'apikey'      => $apiKey,
                'permissions' => ['allow_join', 'allow_mod'],
                'version'     => 2,
                'iat'         => time(),
                //'exp'         => time() + (60 * 60 * 4), // 4 hours
                'exp' => time() + (60 * 60 * 24 * 7),
                //'exp' => time() + 604800,  // ✅ 7 din ka expiry
            ];

            $token = JWT::encode($payload, $secret, 'HS256');
        } catch (\Exception $e) {
            Log::error('JWT generation failed', ['message' => $e->getMessage()]);
            return response()->json([
                'error' => 'Token generation error',
                'message' => $e->getMessage()
            ], 500);
        }

        // ✅ Request validation (meeting_id अब client से नहीं आएगा)
        $request->validate([
            'appointment_id' => 'required|integer',
            'doctor_id'      => 'required|integer',
            'patient_id'     => 'required|integer',
        ]);

        // ✅ Appointment record find karo
        $appointment = Appointment::where('id', $request->appointment_id)
            ->where('doctor_id', $request->doctor_id)
            ->where('patient_id', $request->patient_id)
            ->first();

        if (!$appointment) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Appointment not found',
            ], 404);
        }

        // ✅ Agar meeting_id already hai → wahi return karo
        if (!empty($appointment->meeting_id)) {
            return response()->json([
                'status'  => 'join',
                'message' => 'Meeting already exists',
                'data'    => [
                    'appointment_id' => $appointment->id,
                    'meeting_id'     => $appointment->meeting_id,
                    'token'          => $appointment->token ?: $token,
                ],
            ], 200);
        }

        // ✅ Agar meeting_id nahi hai → VideoSDK pe new room banao
        try {
            $resp = Http::withHeaders([
                    'Authorization' => $token,
                    'Content-Type'  => 'application/json',
                ])
                ->timeout(20)
                ->post('https://api.videosdk.live/v2/rooms', []);

            if (!$resp->ok()) {
                return response()->json([
                    'error'  => 'VideoSDK create room failed',
                    'status' => $resp->status(),
                    'raw'    => $resp->json(),
                ], 500);
            }

            $data       = $resp->json();
            $meeting_id = $data['roomId'] ?? $data['id'] ?? null;

            if (!$meeting_id) {
                return response()->json([
                    'error' => 'Invalid VideoSDK response',
                    'raw'   => $data,
                ], 500);
            }

        } catch (\Throwable $e) {
            Log::error('VideoSDK room create error', ['message' => $e->getMessage()]);
            return response()->json([
                'error'   => 'Unexpected error creating VideoSDK room',
                'message' => $e->getMessage(),
            ], 500);
        }

        // ✅ meeting_id aur token update karo
        $appointment->meeting_id = $meeting_id;
        $appointment->token = $token;
        $appointment->save();

        return response()->json([
            'status'  => 'success',
            'message' => 'Meeting ID and token created successfully',
            'data'    => [
                'appointment_id' => $appointment->id,
                'meeting_id'     => $appointment->meeting_id,
                'token'          => $appointment->token,
            ],
        ], 200);
    }

}
