<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TermCondition; // ✅ ADD THIS LINE
use App\Models\PrivacyPolicy;
use App\Models\Faq;
use App\Models\Notification;
use App\Models\Specialization;
use Illuminate\Http\Request; // ✅ Correct request class

class SettingController extends Controller
{
    // Show all specializations
    public function specializationIndex() {
        $specializations = Specialization::latest()->paginate(10);
        return view('admin.setting.specialization-index', compact('specializations'));
    }

    // Store new specialization
    public function specializationStore(Request $request) {
        $request->validate([
            'specialization' => 'required|string|max:255',
        ]);

        Specialization::create([
            'specialization' => $request->specialization,
            'status' => 1
        ]);

        return redirect()->back()->with('success', 'Specialization added successfully!');
    }

    // Edit form data (AJAX)
    public function specializationEdit($id) {
        $specialization = Specialization::findOrFail($id);
        return response()->json($specialization);
    }

    // Update specialization
    public function specializationUpdate(Request $request, $id) {
        $request->validate([
            'specialization' => 'required|string|max:255',
        ]);

        $specialization = Specialization::findOrFail($id);
        $specialization->update([
            'specialization' => $request->specialization,
        ]);

        return redirect()->back()->with('success', 'Specialization updated successfully!');
    }

    // Delete specialization
    public function specializationDelete($id) {
        Specialization::destroy($id);
        return redirect()->back()->with('success', 'Specialization deleted successfully!');
    }

    // Toggle status
    public function specializationToggle($id) {
        $specialization = Specialization::findOrFail($id);
        $specialization->status = !$specialization->status;
        $specialization->save();

        return redirect()->back()->with('success', 'Status updated successfully!');
    }


    

    // Patient Terms and Conditions
    public function terms()
    {
        $terms = TermCondition::latest()->paginate(1000);
        return view('admin.setting.patient-terms', compact('terms'));
    }

    public function storeTerm(Request $request)
    {
        $request->validate([
            'content' => 'required|string'
        ]);

        TermCondition::create([
            'content' => $request->content,
            'status' => 'active'
        ]);

        return redirect()->route('admin.patient-terms')->with('success', 'Term added successfully.');
    }

    public function editTerm($id)
    {
        $term = TermCondition::findOrFail($id);
        return view('admin.setting.edit_term', compact('term'));
    }

    public function updateTerm(Request $request, $id)
    {
        $request->validate([
            'content' => 'required|string'
        ]);

        $term = TermCondition::findOrFail($id);
        $term->update([
            'content' => $request->content
        ]);

        return redirect()->route('admin.patient-terms')->with('success', 'Term updated.');
    }

    public function deleteTerm($id)
    {
        TermCondition::findOrFail($id)->delete();
        return redirect()->route('admin.patient-terms')->with('success', 'Term deleted.');
    }

    public function toggleStatus($id)
    {
        $term = TermCondition::findOrFail($id);
        $term->status = $term->status === 'active' ? 'inactive' : 'active';
        $term->save();

        return redirect()->route('admin.patient-terms')->with('success', 'Status updated.');
    }


    // Doctor Terms and Conditions
    public function dterms()
    {
        $terms = TermCondition::latest()->paginate(1000);
        return view('admin.setting.doctor-terms', compact('terms'));
    }

    public function dstoreTerm(Request $request)
    {
        $request->validate([
            'content' => 'required|string'
        ]);

        TermCondition::create([
            'content' => $request->content,
            'status' => 'active'
        ]);

        return redirect()->route('admin.doctor-terms')->with('success', 'Term added successfully.');
    }

    public function ddeleteTerm($id)
    {
        TermCondition::findOrFail($id)->delete();
        return redirect()->route('admin.doctor-terms')->with('success', 'Term deleted.');
    }

    public function dtoggleStatus($id)
    {
        $term = TermCondition::findOrFail($id);
        $term->status = $term->status === 'active' ? 'inactive' : 'active';
        $term->save();

        return redirect()->route('admin.doctor-terms')->with('success', 'Status updated.');
    }

    // Patient Privacy Policy
    public function privacy()
    {
        $policies = PrivacyPolicy::orderBy('id', 'desc')->paginate(1000);
        return view('admin.setting.patient-privacy', compact('policies'));
    }

    public function storePrivacy(Request $request)
    {
        $request->validate(['content' => 'required']);
        PrivacyPolicy::create(['content' => $request->content]);
        return back()->with('success', 'Privacy policy added.');
    }

    public function editPrivacy($id)
    {
        $policy = PrivacyPolicy::findOrFail($id);
        return view('admin.setting.edit_privacy', compact('policy'));
    }

    // public function updatePrivacy(Request $request, $id)
    // {
    //     $request->validate(['content' => 'required|string']);
    //     $policy = PrivacyPolicy::findOrFail($id);
    //     $policy->update(['content' => $request->content]);
    //     return redirect()->route('admin.privacy')->with('success', 'Privacy policy updated.');
    // }

    public function updatePrivacy(Request $request, $id)
    {
        $request->validate([
            'content' => 'required|string',
            'status' => 'required|in:active,inactive',
        ]);

        $policy = PrivacyPolicy::findOrFail($id);
        $policy->update([
            'content' => $request->content,
            'status' => $request->status,
        ]);

        return redirect()->route('admin.patient-privacy')->with('success', 'Privacy policy updated.');
    }

    public function deletePrivacy($id)
    {
        PrivacyPolicy::destroy($id);
        return back()->with('success', 'Privacy policy deleted.');
    }

    public function togglePrivacyStatus($id)
    {
        $policy = PrivacyPolicy::findOrFail($id);
        $policy->status = $policy->status === 'active' ? 'inactive' : 'active';
        $policy->save();
        return back()->with('success', 'Status changed.');
    }

    // Doctor Privacy Policy
    public function dprivacy()
    {
        $policies = PrivacyPolicy::orderBy('id', 'desc')->paginate(1000);
        return view('admin.setting.doctor-privacy', compact('policies'));
    }

    public function dstorePrivacy(Request $request)
    {
        $request->validate(['content' => 'required']);
        PrivacyPolicy::create(['content' => $request->content]);
        return back()->with('success', 'Privacy policy added.');
    }

    public function ddeletePrivacy($id)
    {
        PrivacyPolicy::destroy($id);
        return back()->with('success', 'Privacy policy deleted.');
    }

    public function dtogglePrivacyStatus($id)
    {
        $policy = PrivacyPolicy::findOrFail($id);
        $policy->status = $policy->status === 'active' ? 'inactive' : 'active';
        $policy->save();
        return back()->with('success', 'Status changed.');
    }

    // Patient FAQ
    public function faq()
    {
        $faqs = Faq::latest()->get();
        return view('admin.setting.patient-faq', compact('faqs'));
    }

    public function storeFaq(Request $request)
    {
        $request->validate([
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
        ]);

        Faq::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'status' => 'active',
        ]);

        return redirect()->route('admin.patient-faq')->with('success', 'FAQ added successfully.');
    }

    public function editFaq($id)
    {
        $faq = Faq::findOrFail($id);
        return view('admin.setting.edit_faq', compact('faq'));
    }

    public function updateFaq(Request $request, $id)
    {
        $request->validate([
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
        ]);

        $faq = Faq::findOrFail($id);
        $faq->update([
            'question' => $request->question,
            'answer' => $request->answer,
        ]);

        return redirect()->route('admin.patient-faq')->with('success', 'FAQ updated successfully.');
    }

    public function destroyFaq($id)
    {
        Faq::destroy($id);
        return redirect()->route('admin.patient-faq')->with('success', 'FAQ deleted.');
    }

    public function toggleFaqStatus($id)
    {
        $faq = Faq::findOrFail($id);
        $faq->status = $faq->status === 'active' ? 'inactive' : 'active';
        $faq->save();

        return redirect()->route('admin.patient-faq')->with('success', 'FAQ status updated.');
    }


    // Doctor FAQ
    public function dfaq()
    {
        $faqs = Faq::latest()->get();
        return view('admin.setting.doctor-faq', compact('faqs'));
    }

    public function dstoreFaq(Request $request)
    {
        $request->validate([
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
        ]);

        Faq::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'status' => 'active',
        ]);

        return redirect()->route('admin.doctor-faq')->with('success', 'FAQ added successfully.');
    }

    public function ddestroyFaq($id)
    {
        Faq::destroy($id);
        return redirect()->route('admin.doctor-faq')->with('success', 'FAQ deleted.');
    }

    public function dtoggleFaqStatus($id)
    {
        $faq = Faq::findOrFail($id);
        $faq->status = $faq->status === 'active' ? 'inactive' : 'active';
        $faq->save();

        return redirect()->route('admin.doctor-faq')->with('success', 'FAQ status updated.');
    }

    // Notification
    public function notification()
    {
        $notifications = Notification::latest()->get();
        return view('admin.setting.notification', compact('notifications'));
    }

    public function storeNotification(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'message' => 'required|string',
        ]);

        Notification::create([
            'name' => $request->name,
            'title' => $request->title,
            'message' => $request->message,
        ]);

        return redirect()->route('admin.notification')->with('success', 'Notification added successfully.');
    }

    public function destroyNotification($id)
    {
        $notification = Notification::findOrFail($id);
        $notification->delete();

        return redirect()->route('admin.notification')->with('success', 'Notification deleted successfully.');
    }
}

//return view('livewire.admin.setting.terms', compact('terms'));



