<?php
namespace App\Http\Controllers\Admin;

use App\Models\User;
use Laravel\Sanctum\Guard;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class CustomAuthController extends Controller
{
    public function index()
    {
        return view('Pages.login');
    }

    public function customLogin(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            $user = Auth::user();

            // ✅ Check if user has permission to access any dashboard
            if (
                $user->hasAnyPermission([
                    'access superadmin dashboard',
                    'access doctor dashboard',
                    'access patient dashboard',
                ])
            ) {
                notyf()->success('Login successful!');
                return redirect()->intended(route('admin.dashboard'));
            }

            Auth::logout();
            return redirect()->route('admin.login')->withErrors([
                'email' => 'You are not authorized to access this system.'
            ]);
        }


        return back()->withErrors([
            'email' => 'The provided credentials do not match our records.',
        ]);
    }

    public function registration()
    {
        return view('Pages.register');
    }

    public function customRegistration(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'role' => 'required|in:doctor,patient',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        $user->syncRoles([$request->role]);

        notyf()->success('Registration completed! Please login.');
        return redirect()->route('admin.login');
    }

    public function dashboard()
    {
        $guardLoginRoutes = [
            'superadmin' => 'admin.login',
            'doctor' => 'doctor.login',
            'staff' => 'staff.login',
            'patient' => 'patient.login',
        ];
        $guards = ['superadmin', 'doctor', 'staff', 'patient'];

        foreach ($guards as $guard) {
            if (Auth::guard($guard)->check()) {
                $user = Auth::guard($guard)->user();

                $permissions = $user->getAllPermissions()->pluck('name')->toArray();

                if ($guard === 'superadmin' && in_array('access superadmin dashboard', $permissions)) {
                    return view('Dashboard.index');
                } elseif ($guard === 'doctor' && in_array('access doctor dashboard', $permissions)) {
                    return redirect()->route('admin.doctor-dashboard');
                } elseif ($guard === 'patient' && in_array('access patient dashboard', $permissions)) {
                    return redirect()->route('admin.patient-dashboard');
                } elseif ($guard === 'staff' && in_array('access staff dashboard', $permissions)) {
                    return redirect()->route('admin.staff-dashboard');
                }

                Auth::guard($guard)->logout();
                return redirect()->route($guardLoginRoutes[$guard])
                    ->withErrors(['unauthorized' => 'You do not have permission to access any dashboard.']);
            }
        }

        Log::info("No authenticated guard found when accessing dashboard.");
        return redirect()->route('login')->withErrors([
            'email' => 'Please log in to access the system.'
        ]);
    }

    function profile()
    {
        return view('admin.profile.admin-profile');
    }

    public function signOut()
    {
        Session::flush();
        Auth::logout();

        notyf()->success('You have been logged out successfully.');
        return redirect()->route('admin.login');
    }
}