<?php

namespace App\Http\Controllers\Admin\Api\V1;

use App\Models\User;
use App\Models\Clinic;
use App\Models\Doctor;
use App\Models\DoctorClinic;
use App\Models\DoctorProfile;
use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use App\Models\CustomerSubscription;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class ClinicApiController extends Controller
{
    public function index(Request $request)
    {
        $clinics = Clinic::with(['owner', 'subscription'])
            ->when($request->search, function ($query, $search) {
                return $query->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            })
            ->when($request->status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'status' => 'success',
            'data' => $clinics
        ]);
    }

    public function store(Request $request)
    {
        $this->validateRequest($request);

        try {
            DB::beginTransaction();

            $clinicData = $this->prepareClinicData($request);
            $clinic = Clinic::create($clinicData);

            if ($request->owner_id) {
                $this->handleDoctorClinicAssociation($clinic, $request->owner_id);
                $this->handleSubscription($clinic, $request);
            }

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Clinic created successfully',
                'data' => $clinic->load(['owner', 'subscription'])
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Clinic creation error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create clinic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show(Clinic $clinic)
    {
        return response()->json([
            'status' => 'success',
            'data' => $clinic->load(['owner', 'subscription', 'doctors'])
        ]);
    }

    public function update(Request $request, Clinic $clinic)
    {
        $this->validateRequest($request, $clinic);

        try {
            DB::beginTransaction();

            $clinicData = $this->prepareClinicData($request);
            $clinic->update($clinicData);

            if ($request->owner_id) {
                $this->handleDoctorClinicAssociation($clinic, $request->owner_id);
                $this->handleSubscription($clinic, $request);
            } else {
                if ($clinic->subscription_id) {
                    CustomerSubscription::where('id', $clinic->subscription_id)->delete();
                    $clinic->update(['subscription_id' => null]);
                }
            }

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Clinic updated successfully',
                'data' => $clinic->load(['owner', 'subscription'])
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Clinic update error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update clinic',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy(Clinic $clinic)
    {
        try {
            $clinic->delete();
            return response()->json([
                'status' => 'success',
                'message' => 'Clinic deleted successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Clinic deletion error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete clinic'
            ], 500);
        }
    }

    public function createDoctor(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'specialization' => 'required|string|max:255',
            'license_number' => 'required|string|unique:doctors,license_number',
        ]);

        try {
            DB::beginTransaction();

            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'status' => 'active',
            ]);

            $doctorRoleWeb = Role::where('name', 'primary_doctor')->where('guard_name', 'web')->first();
            $doctorRoleDoctor = Role::where('name', 'primary_doctor')->where('guard_name', 'doctor')->first();

            if ($doctorRoleWeb) {
                $user->assignRole($doctorRoleWeb);
            }

            if ($doctorRoleDoctor) {
                $user->assignRole($doctorRoleDoctor);
            }

            $doctor = Doctor::create([
                'user_id' => $user->id,
                'specialization' => $request->specialization,
                'license_number' => $request->license_number,
            ]);

            DoctorProfile::create([
                'user_id' => $user->id,
                'specialization' => $request->specialization,
                'license_number' => $request->license_number,
            ]);

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Doctor created successfully',
                'data' => $user
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Doctor creation error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create doctor',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function availableOwners()
    {
        $owners = User::role('doctor')
            ->whereDoesntHave('clinicAssociations', function ($query) {
                $query->where('is_primary', true);
            })
            ->with('doctorProfile')
            ->get();

        return response()->json([
            'status' => 'success',
            'data' => $owners
        ]);
    }



    public function availablePlans()
    {
        $plans = SubscriptionPlan::where('plan_status', 'active')->get();
        return response()->json([
            'status' => 'success',
            'data' => $plans
        ]);
    }

    protected function validateRequest(Request $request, Clinic $clinic = null)
    {
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'owner_id' => 'nullable|exists:users,id',
            'about' => 'nullable|string',
            'line1' => 'nullable|string|max:255',
            'line2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'country' => 'nullable|string|max:100',
            'pincode' => 'nullable|string|max:20',
            'location' => 'nullable',
            'specialities' => 'nullable|array',
            'contact_numbers' => 'nullable|array',
            'status' => 'required|in:active,inactive',
            'selected_plan_id' => 'nullable|exists:subscription_plans,id',
        ];

        if ($clinic) {
            $rules['email'] .= '|unique:clinics,email,' . $clinic->id;
        } else {
            $rules['email'] .= '|unique:clinics,email';
        }

        $request->validate($rules);

        if ($request->owner_id) {
            $alreadyPrimary = DoctorClinic::where('doctor_id', $request->owner_id)
                ->where('is_primary', true)
                ->when($clinic?->id, fn($q) => $q->where('clinic_id', '!=', $clinic->id))
                ->exists();

            if ($alreadyPrimary) {
                throw ValidationException::withMessages([
                    'owner_id' => ['This doctor is already a primary doctor in another clinic.']
                ]);
            }
        }
    }

    protected function prepareClinicData(Request $request)
    {
        return [
            'name' => $request->name,
            'email' => $request->email,
            'owner_id' => $request->owner_id,
            'about' => $request->about,
            'line1' => $request->line1,
            'line2' => $request->line2,
            'city' => $request->city,
            'state' => $request->state,
            'country' => $request->country,
            'pincode' => $request->pincode,
            'location' => $request->location,
            'specialities' => $request->specialities ?? null,
            'contact_numbers' => $request->contact_numbers ?? null,
            'status' => $request->status,
            'added_by' => auth()->id(),
        ];
    }

    protected function handleDoctorClinicAssociation(Clinic $clinic, $doctorId)
    {
        DoctorClinic::updateOrCreate(
            ['doctor_id' => $doctorId, 'clinic_id' => $clinic->id],
            [
                'is_primary' => true,
                'status' => 'active',
                'approval_status' => 'approved',
                'approved_at' => now(),
                'offers_in_person' => true,
                'offers_video' => true,
                'offers_home_visit' => false,
                'added_by' => auth()->id(),
            ]
        );
    }

    protected function handleSubscription(Clinic $clinic, Request $request)
    {
        if (!$request->selected_plan_id) {
            if ($clinic->subscription_id) {
                $clinic->update(['subscription_id' => null]);
                CustomerSubscription::where('id', $clinic->subscription_id)->delete();
            }
            return;
        }

        $plan = SubscriptionPlan::findOrFail($request->selected_plan_id);

        if ($clinic->subscription_id) {
            $currentSubscription = CustomerSubscription::find($clinic->subscription_id);

            if (!$currentSubscription || $currentSubscription->plan_id != $request->selected_plan_id) {
                $clinic->update(['subscription_id' => null]);
                if ($currentSubscription) {
                    $currentSubscription->delete();
                }
                $this->createNewSubscription($clinic, $plan, $request->owner_id);
            } else {
                $currentSubscription->update([
                    'user_id' => $request->owner_id,
                    'status' => 'active',
                    'auto_renew' => true,
                ]);
            }
        } else {
            $this->createNewSubscription($clinic, $plan, $request->owner_id);
        }
    }

    protected function createNewSubscription(Clinic $clinic, SubscriptionPlan $plan, $ownerId)
    {
        $subscription = CustomerSubscription::create([
            'subscription_code' => 'SUB-' . strtoupper(uniqid()),
            'user_id' => $ownerId,
            'plan_id' => $plan->id,
            'clinic_id' => $clinic->id,
            'start_date' => now(),
            'end_date' => now()->addMonths($plan->plan_type === 'annual' ? 12 : 1),
            'renewal_date' => now()->addMonths($plan->plan_type === 'annual' ? 12 : 1),
            'status' => 'active',
            'auto_renew' => true,
            'billing_cycle' => $plan->plan_type,
        ]);

        $clinic->update(['subscription_id' => $subscription->id]);
    }
}