<?php

namespace App\Services;

use App\Models\User;
use App\Models\StaffClinic;
use Illuminate\Support\Facades\Log;

class StaffClinicService
{
    public function getAvailableDoctorClinicPairs(User $user): array
    {
        if (!$user->isStaff()) {
            Log::info("User {$user->id} is not staff. Returning empty list.");
            return [];
        }

        return StaffClinic::with(['doctor.roles', 'clinic'])
            ->where('staff_id', $user->id)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->get()
            ->map(function ($association) {
                return $this->formatDoctorClinicData($association);
            })
            ->sortByDesc(function ($pair) {
                return $pair['is_primary'] ? 1 : 0;
            })
            ->values()
            ->toArray();
    }

    public function getCurrentDoctorClinicPair(User $user): ?array
    {
        if (!$user->isStaff()) {
            return null;
        }

        $doctorId = session('current_doctor_id');
        $clinicId = session('current_clinic_id');

        if ($doctorId && $clinicId) {
            $association = $this->getClinicAssociation($user, $doctorId, $clinicId);
            if ($association) {
                return $association;
            }
        }

        $first = $this->getFirstAvailablePair($user);
        if ($first) {
            $this->setCurrentDoctorClinicInSession($first['doctor_id'], $first['clinic_id']);
            return $first;
        }

        return null;
    }
    public function checkPermissionForPair(User $user, int $doctorId, int $clinicId, string $permission): bool
    {
        $association = $this->getClinicAssociation($user, $doctorId, $clinicId);

        if (!$association) {
            return false;
        }

        return in_array($permission, $association['permissions']);
    }
    public function getClinicAssociation(User $user, int $doctorId, int $clinicId): ?array
    {
        $association = StaffClinic::with(['doctor.roles', 'clinic'])
            ->where('staff_id', $user->id)
            ->where('doctor_id', $doctorId)
            ->where('clinic_id', $clinicId)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->first();

        return $association ? $this->formatDoctorClinicData($association) : null;
    }

    protected function formatDoctorClinicData(StaffClinic $association): array
    {
        $permissions = $this->parsePermissions($association->permissions);

        return [
            'doctor_id' => $association->doctor_id,
            'doctor_name' => $association->doctor->name,
            'clinic_id' => $association->clinic_id,
            'clinic_name' => $association->clinic->name,
            'association_id' => $association->id,
            'role_type' => $this->determineRoleType($association),
            'permissions' => $permissions,
            'permissions_count' => count($permissions),
            'is_primary' => $association->is_primary,
            'doctor_roles' => $association->doctor->roles->pluck('name')->toArray()
        ];
    }

    protected function parsePermissions($permissions)
    {
        if (is_string($permissions)) {
            try {
                $decoded = json_decode($permissions, true);
                return is_array($decoded) ? $decoded : [];
            } catch (\Exception $e) {
                return [];
            }
        }
        return is_array($permissions) ? $permissions : [];
    }

    protected function determineRoleType(StaffClinic $association): string
    {
        $doctor = $association->doctor;

        if ($doctor->hasRole('primary_doctor')) {
            return 'primary_doctor';
        }
        if ($doctor->hasRole('sub_doctor')) {
            return 'sub_doctor';
        }
        if ($doctor->hasRole('doctor')) {
            return 'doctor';
        }
        return 'staff';
    }

    public function setCurrentDoctorClinicInSession(int $doctorId, int $clinicId): void
    {
        session([
            'current_doctor_id' => $doctorId,
            'current_clinic_id' => $clinicId
        ]);
    }

    protected function getFirstAvailablePair(User $user): ?array
    {
        $association = StaffClinic::with(['doctor.roles', 'clinic'])
            ->where('staff_id', $user->id)
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->orderBy('is_primary', 'desc')
            ->first();

        return $association ? $this->formatDoctorClinicData($association) : null;
    }
}