<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable, HasRoles, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'phone_number',
        'email',
        'password',
        'status',
        'max_active_devices',
        'last_login_at'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'last_login_at' => 'datetime',
        ];
    }
    public function ownedClinics()
    {
        return $this->hasOne(Clinic::class, 'owner_id');
    }
    public function isClinicOwner()
    {
        return $this->ownedClinics()->exists();
    }
    public function doctorProfile()
    {
        return $this->hasOne(DoctorProfile::class , 'user_id');
    }

    public function clinics()
    {
        return $this->belongsToMany(Clinic::class, 'doctor_clinic', 'doctor_id', 'clinic_id')
            ->using(DoctorClinic::class)
            ->withPivot([
                'is_primary',
                'status',
                'in_person_fee',
                'video_fee',
                'home_visit_fee',
                'added_by',
                'subscription_member_id'
            ]);
    }

    public function primaryClinics()
    {
        return $this->clinics()->wherePivot('is_primary', true);
    }

    public function isPrimaryAtAnyClinic()
    {
        return $this->primaryClinics()->exists();
    }

    public function clinicAssociations()
    {
        return $this->hasMany(DoctorClinic::class, 'doctor_id');
    }
    public function staffClinicAssociations()
    {
        return $this->hasMany(StaffClinic::class, 'staff_id');
    }

    public function activeClinicAssociations()
    {
        $activeClinics = $this->clinicAssociations()
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->get();
        return $activeClinics->count();
    }
    public function activeStaffClinicAssociations()
    {
        $activeClinics = $this->staffClinicAssociations()
            ->where('approval_status', 'approved')
            ->where('status', 'active')
            ->get();
        return $activeClinics->count();
    }

    public function isPrimaryAtClinic(Clinic $clinic)
    {
        return $this->primaryClinics()->where('clinics.id', $clinic->id)->exists();
    }

    public function subscriptionMemberships()
    {
        return $this->hasMany(DoctorSubscriptionMember::class, 'doctor_id');
    }

    public function addedDoctors()
    {
        return $this->hasMany(DoctorClinic::class, 'added_by');
    }

    public function patient()
    {
        return $this->hasOne(Patient::class);
    }

    public function getRedirectRoute()
    {
        return match (true) {
            $this->hasRole('super-admin') => 'admin/dashboard',
            $this->hasRole('doctor') => 'doctor-dashboard',
            $this->hasRole('patient') => 'patient-dashboard',
            default => 'dashboard',
        };
    }

    public function getActiveSubscription()
    {
        return $this->subscriptionMemberships()
            ->whereHas('subscription', function ($q) {
                $q->where('status', 'active');
            })
            ->first();
    }

    public function canAddDoctorToClinic(Clinic $clinic)
    {
        if ($this->isPrimaryAtClinic($clinic)) {
            return $clinic->hasActiveSubscription();
        }

        return false;
    }

    public function activeSubscription()
    {
        return $this->hasOne(CustomerSubscription::class, 'user_id')
            ->where('status', 'active')
            ->with('plan')
            ->latest();
    }
    public function doctor()
    {
        return $this->hasOne(Doctor::class);
    }
    public function isDoctor()
    {
        return $this->hasAnyRole(['doctor', 'primary_doctor', 'sub_doctor']);
    }
    public function isStaff()
    {
        return $this->hasRole('doctorstaff');
    }
    public function staffClinics()
    {
        return $this->hasMany(StaffClinic::class, 'staff_id')
            ->where('approval_status', 'approved');
    }

    public function patientClinics()
    {
        return $this->hasMany(PatientClinic::class, 'patient_id')
            ->where('approval_status', 'approved');
    }
    public function doctorClinics()
    {
        return $this->belongsToMany(Clinic::class, 'doctor_clinic', 'doctor_id', 'clinic_id')
            ->withPivot(['override_default_permissions', 'extra_permissions']);
    }
    public function managedClinics()
    {
        return $this->belongsToMany(Clinic::class, 'staff_clinics', 'staff_id', 'clinic_id');
    }

    public function supervisingDoctors()
    {
        return $this->belongsToMany(User::class, 'staff_clinics', 'staff_id', 'doctor_id')
            ->wherePivot('approval_status', 'approved')
            ->withPivot(['clinic_id', 'permissions']);
    }

    public function getCurrentDoctorClinicPair()
    {
        $doctorId = session('current_doctor_id');
        $clinicId = session('current_clinic_id');

        if ($doctorId && $clinicId) {
            return [
                'doctor' => User::find($doctorId),
                'clinic' => Clinic::find($clinicId)
            ];
        }
        return null;
    }

    public function hasPrimaryClinic(): bool
    {
        return DoctorClinic::where('doctor_id', $this->id)
            ->where('is_primary', true)
            ->where('approval_status', 'approved')
            ->exists();
    }

    public function profile()
    {
        return $this->hasOne(UserProfile::class);
    }

    public function getProfilePhotoUrlAttribute()
    {
        if ($this->doctor && $this->doctor->profile_image) {
            if (filter_var($this->doctor->profile_image, FILTER_VALIDATE_URL)) {
                return $this->doctor->profile_image;
            }

            return asset('storage/' . $this->doctor->profile_image);
        }

        if ($this->profile_image) {
            if (filter_var($this->profile_image, FILTER_VALIDATE_URL)) {
                return $this->profile_image;
            }
            return asset('storage/' . $this->profile_image);
        }

        return null;
    }

    public function getInitialsAvatar()
    {
        $nameParts = explode(' ', $this->name);
        $initials = '';

        foreach ($nameParts as $part) {
            $initials .= strtoupper(substr($part, 0, 1));
        }

        $initials = substr($initials, 0, 2);

        return '<div class="avatar-initials" style="width: 120px; height: 120px; background: linear-gradient(135deg, #1976D2 0%, #0D47A1 100%); color: white; display: flex; align-items: center; justify-content: center; border-radius: 4px; font-weight: bold; font-size: 40px;">' . $initials . '</div>';
    }
    public function staff()
    {
        return $this->hasOne(Staff::class);
    }

}
