<?php

namespace App\Livewire\User;

use Livewire\Component;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use Illuminate\Support\Carbon;

class ResetPasswordForm extends Component
{
    public $token;
    public $email;
    public $password;
    public $password_confirmation;
    public $status = '';
    public $disable = false;

    public function mount($token, $email)
    {
        $this->token = $token;
        $this->email = $email;
    }

    protected $rules = [
        'password' => 'required|min:8|confirmed',
    ];

    public function resetPassword()
    {
        $this->validate();
        $this->disable = true;

        try {
            // Check if token is valid and not expired (1 hour expiration)
            $record = DB::table('password_reset_tokens')
                        ->where('email', $this->email)
                        ->first();

            // Check if token exists
            if (!$record) {
                notyf()->error('Invalid or expired token !');
                throw new \Exception('Invalid or expired token.');
            }

            // Check if token matches (hashed comparison)
            if (!hash_equals($record->token, hash('sha256', $this->token))) {
                notyf()->error('Invalid Token !');
                throw new \Exception('Invalid token.');
            }

            // Check if token is expired
            if (Carbon::parse($record->created_at)->addMinutes(30)->isPast()) {
                notyf()->error('This password reset link has expired !');
                throw new \Exception('This password reset link has expired.');
            }

            // Update user's password
            $user = User::where('email', $this->email)->first();
            $user->password = Hash::make($this->password);
            $user->save();

            // Delete the token
            DB::table('password_reset_tokens')->where('email', $this->email)->delete();

            notyf()->success('Your password has been reset successfully!');
            $this->status = 'Your password has been reset successfully!';
            $this->dispatch('password-reset-success');
            
        } catch (\Exception $e) {
            $this->status = $e->getMessage();
            $this->disable = false;
        }
    }

    public function render()
    {
        return view('livewire.user.reset-password-form');
    }
}