<?php

namespace App\Livewire\Patient\Profile;

use App\Models\Patient;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Livewire\Component;
use Livewire\WithFileUploads;

class PatientProfile extends Component
{
    use WithFileUploads;

    public $name;
    public $email;
    public $phone_number;

    public $address;
    public $street_address;
    public $city;
    public $state;
    public $postal_code;
    public $country;
    public $phone;
    public $date_of_birth;
    public $gender;
    public $medical_history;
    public $profile_image;
    public $new_profile_image;

    public $isEditing = false;
    public $patient;
    public $user;
    protected $listeners = ['dateChanged'];

    public function dateChanged($value)
    {
        $this->date_of_birth = $value;
        // Optional: Add validation or logging here
        //logger()->info('Date of birth updated to: ' . $value);
    }

    protected function rules()
    {
        return [
            'name' => 'required|string|max:255',
            'email' => ['required', 'email', 'max:255', Rule::unique('users', 'email')->ignore(Auth::id())],
            'phone_number' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:500',
            'street_address' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|in:male,female,other',
            'medical_history' => 'nullable|string',
            'new_profile_image' => 'nullable|image|max:2048',
        ];
    }

    protected $validationAttributes = [
        'name' => 'full name',
        'phone_number' => 'phone number',
        'street_address' => 'street address',
        'postal_code' => 'postal code',
        'date_of_birth' => 'date of birth',
        'medical_history' => 'medical history',
        'new_profile_image' => 'profile image',
    ];

    public function mount()
    {
        try {
            $this->user = Auth::user();
            $this->patient = $this->user->patient ?? new Patient();

            // Load user data
            $this->name = $this->user->name;
            $this->email = $this->user->email;
            $this->phone_number = $this->user->phone_number;

            // Load patient data
            if ($this->patient->exists) {
                $this->address = $this->patient->address;
                $this->street_address = $this->patient->street_address;
                $this->city = $this->patient->city;
                $this->state = $this->patient->state;
                $this->postal_code = $this->patient->postal_code;
                $this->country = $this->patient->country;
                $this->phone = $this->patient->phone;
                $this->date_of_birth = $this->patient->getDateOfBirthFormatted();
                $this->gender = $this->patient->gender;
                $this->medical_history = $this->patient->medical_history;
                $this->profile_image = $this->patient->profile_image;
            }

        } catch (\Exception $e) {
            notyf()->error('Failed to load profile data. Please try again.');
        }
    }

    public function toggleEdit()
    {
        $this->isEditing = !$this->isEditing;
        
        if (!$this->isEditing) {
            $this->mount();
            $this->resetValidation();
            notyf()->info('Edit mode canceled. Changes discarded.');
        } else {
            notyf()->info('Now in edit mode. You can update your profile.');
        }
    }

    public function save()
    {
        $this->validate();

        try {
            DB::beginTransaction();

            $this->user->update([
                'name' => $this->name,
                'email' => $this->email,
                'phone_number' => $this->phone_number,
            ]);

            $profileImagePath = $this->profile_image;
            if ($this->new_profile_image) {
                if ($this->profile_image && Storage::disk('public')->exists($this->profile_image)) {
                    Storage::disk('public')->delete($this->profile_image);
                }
                
                $filename = 'patient-' . $this->user->id . '-' . Str::random(10) . '.' . $this->new_profile_image->extension();
                $profileImagePath = $this->new_profile_image->storeAs('patient-profiles', $filename, 'public');
            }

            $patientData = [
                'user_id' => $this->user->id,
                'address' => $this->address,
                'street_address' => $this->street_address,
                'city' => $this->city,
                'state' => $this->state,
                'postal_code' => $this->postal_code,
                'country' => $this->country,
                'phone' => $this->phone,
                'date_of_birth' => $this->date_of_birth,
                'gender' => $this->gender,
                'medical_history' => $this->medical_history,
                'profile_image' => $profileImagePath,
            ];

            if ($this->patient->exists) {
                $this->patient->update($patientData);
            } else {
                $this->patient = Patient::create($patientData);
            }

            $this->profile_image = $profileImagePath;
            $this->new_profile_image = null;
            $this->isEditing = false;

            DB::commit();

            notyf()->success('Profile updated successfully!');
            
        } catch (\Exception $e) {
            DB::rollBack();
            notyf()->error('Failed to update profile. Please try again.');
        }
    }

    public function removeProfileImage()
    {
        try {
            if ($this->profile_image && Storage::disk('public')->exists($this->profile_image)) {
                Storage::disk('public')->delete($this->profile_image);
            }

            if ($this->patient->exists) {
                $this->patient->update(['profile_image' => null]);
            }

            $this->profile_image = null;
            $this->new_profile_image = null;
            
            notyf()->success('Profile image removed successfully!');
        } catch (\Exception $e) {
            notyf()->error('Failed to remove profile image. Please try again.');
        }
    }

    public function getProfileImageUrl()
    {
        try {
            if ($this->new_profile_image) {
                return $this->new_profile_image->temporaryUrl();
            }

            if ($this->profile_image) {
                return Storage::url($this->profile_image);
            }

            return asset('images/default-avatar.png');
        } catch (\Exception $e) {
            return asset('images/default-avatar.png');
        }
    }

    public function render()
    {
        return view('livewire.patient.profile.patient-profile');
    }
}